/*----------------------------------------------------------------------
Pac-Man Evolution - Roberto Prieto
 Copyright (C) 2018-2025 MegaStorm Systems
contact@megastormsystems.com - http://www.megastormsystems.com

This software is provided 'as-is', without any express or implied
warranty.  In no event will the authors be held liable for any damages
arising from the use of this software.

Permission is granted to anyone to use this software for any purpose,
including commercial applications, and to alter it and redistribute it
freely, subject to the following restrictions:

1. The origin of this software must not be misrepresented; you must not
claim that you wrote the original software. If you use this software
in a product, an acknowledgment in the product documentation would be
appreciated but is not required.
2. Altered source versions must be plainly marked as such, and must not be
misrepresented as being the original software.
3. This notice may not be removed or altered from any source distribution.

------------------------------------------------------------------------

Resource Manager

------------------------------------------------------------------------ */

#include "ResourceManager.h"
#include "Pac-Man_Evolution.h"
#include <chrono>

extern unsigned char pme_icon[];
extern int pme_icon_size;

// Singleton stuff
ResourceManager* ResourceManager::mInstance = nullptr;

// Create a single instance
ResourceManager& ResourceManager::Instance()
{
    if(!mInstance) mInstance = new(std::nothrow) ResourceManager;
    return *mInstance;
}

//! Explicit destructor.
void ResourceManager::Terminate()
{
    if(mInstance) delete mInstance;
    mInstance = nullptr;
}

ResourceManager::ResourceManager()
{
    bPacManDeathAnim = 1;
    guiMenu = guiGame = 0;
    fntConsole = fntInfo = fntScore = 0;
    musMenu[0] = musMenu[1] = musMenu[2] = 0;
    musGame[0] = musGame[1] = musGame[2] = musGame[3] = 0;
    sndThunder = sndClickOK = sndClickCancel = sndClickCancel2 = sndExit = 0;
    sndGameAbort = sndGameOver = sndGameStarting = sndGamePlayerDeath = sndGameEatPellet = sndGameEatPelletPower = sndGameEatGhost = 0;
    imgIcon = imgMenu = imgHallOfFame = 0;
    curArrow = 0;
    sprPacman = sprGhostRed = sprGhostPink = sprGhostBlue = sprGhostOrange = sprTargets = 0;
    sprPellet = sprPelletPower = 0;
}

ResourceManager::~ResourceManager()
{
    free();
}

// Load all resources. Avoid double loads and manager errors.
Sint32 ResourceManager::load()
{
    Sint32 bErrorFlag = 0;
    Main& mC64 = Main::Instance();
    ImageMgr& mImage = mC64.IImageMgr();
    SpriteMgr& mSprite = mC64.ISpriteMgr();
    FontMgr& mFont = Main::Instance().IFontMgr();
    AudioTrackMgr& mAudio = Main::Instance().IAudioTrackMgr();
    
    // Load the fonts
    if(fntConsole <= 0)
    {
        fntConsole = mFont.load(PME_RESOURCES, "console");
        if(fntConsole < 0)
        {
            mC64.ILogMgr().get()->msg(LML_CRITICAL, "  [ResourceManager] Critical: could not load the console font.\n");
            bErrorFlag = 1;
        }
    }
    if(fntInfo <= 0 && bErrorFlag == 0)
    {
        fntInfo = mFont.load(PME_RESOURCES, "info");
        if(fntInfo < 0)
        {
            mC64.ILogMgr().get()->msg(LML_CRITICAL, "  [ResourceManager] Critical: could not load the info font.\n");
            bErrorFlag = 1;
        }
    }
    if(fntScore <= 0 && bErrorFlag == 0)
    {
        fntScore = mFont.load(PME_RESOURCES, "score");
        if(fntScore < 0)
        {
            mC64.ILogMgr().get()->msg(LML_CRITICAL, "  [ResourceManager] Critical: could not load the score font.\n");
            bErrorFlag = 1;
        }
    }
    
    // Create the menu panel using sprites stored on the CDC
    if(guiMenu <= 0 && bErrorFlag == 0)
    {
        guiMenu = mC64.IGUIMgr().create("menu");
        if(guiMenu < 0)
        {
            mC64.ILogMgr().get()->msg(LML_CRITICAL, "  [ResourceManager] Critical: could not create the menu panel.\n");
            bErrorFlag = 1;
        }
        else
        {
            Panel* myPanel = mC64.IGUIMgr().getPanel(guiMenu);
            myPanel->baseWidget().setSize(200, 250);
            myPanel->baseWidget().setPosition(PH_CENTER, Position(PH_CENTER, -50.0f));
            myPanel->baseWidget().setBorderWidth(0);
            myPanel->baseWidget().features(WIDGET_FEATURE_BGDISABLE);

            // Add our buttons
            myPanel->createWidget("Exit", WT_BUTTON, ID_EXIT);
            myPanel->getWidget(ID_EXIT)->setKey(SDLK_ESCAPE);
            myPanel->getWidget(ID_EXIT)->setText("");
            myPanel->getWidget(ID_EXIT)->setBgSprite(PME_RESOURCES, "widget_exit");
            myPanel->getWidget(ID_EXIT)->setPosition(PH_CENTER, PH_BOTTOM);

            myPanel->createWidget("StandardGame", WT_BUTTON, ID_STANDARDGAME);
            myPanel->getWidget(ID_STANDARDGAME)->setKey(SDLK_S);
            myPanel->getWidget(ID_STANDARDGAME)->setText("");
            myPanel->getWidget(ID_STANDARDGAME)->setBgSprite(PME_RESOURCES, "widget_game_standard");
            myPanel->getWidget(ID_STANDARDGAME)->setPosition(PH_CENTER, PH_TOP);

            myPanel->createWidget("EvolutionGame", WT_BUTTON, ID_EVOLUTIONGAME);
            myPanel->getWidget(ID_EVOLUTIONGAME)->setKey(SDLK_E);
            myPanel->getWidget(ID_EVOLUTIONGAME)->setText("");
            myPanel->getWidget(ID_EVOLUTIONGAME)->setBgSprite(PME_RESOURCES, "widget_game_evolution");
            myPanel->getWidget(ID_EVOLUTIONGAME)->setPosition(PH_CENTER, Position(PH_TOP, 50.0f));

        #ifdef PME_WORKBENCH_ENABLED
            myPanel->createWidget("Workbench", WT_BUTTON, ID_WORKBENCHGAME);
            myPanel->getWidget(ID_WORKBENCHGAME)->setKey(SDLK_w);
            myPanel->getWidget(ID_WORKBENCHGAME)->setText("");
            myPanel->getWidget(ID_WORKBENCHGAME)->setBgSprite(PME_RESOURCES, "widget_game_workbench");
            myPanel->getWidget(ID_WORKBENCHGAME)->setPosition(PH_CENTER, Position(PH_TOP, 100.0f));
        #endif

            myPanel->createWidget("HoF", WT_BUTTON, ID_HALLOFFAME);
            myPanel->getWidget(ID_HALLOFFAME)->setKey(SDLK_H);
            myPanel->getWidget(ID_HALLOFFAME)->setText("");
            myPanel->getWidget(ID_HALLOFFAME)->setBgSprite(PME_RESOURCES, "widget_hof");
        #ifdef PME_WORKBENCH_ENABLED
            myPanel->getWidget(ID_HALLOFFAME)->setPosition(PH_CENTER, Position(PH_TOP, 150.0f));
        #else
            myPanel->getWidget(ID_HALLOFFAME)->setPosition(PH_CENTER, Position(PH_TOP, 100.0f));
        #endif
        }
    }
    
    // Create the game panel using sprites stored on the CDC
    if(guiGame <= 0 && bErrorFlag == 0)
    {
        guiGame = mC64.IGUIMgr().create("game");
        if(guiGame < 0)
        {
            mC64.ILogMgr().get()->msg(LML_CRITICAL, "  [ResourceManager] Critical: could not create the game panel.\n");
            bErrorFlag = 1;
        }
        else
        {
            Panel* myPanel = mC64.IGUIMgr().getPanel(guiGame);
            myPanel->baseWidget().setSize(300, 250);
            myPanel->baseWidget().setPosition(PH_CENTER, PH_CENTER);
            myPanel->baseWidget().setBorderWidth(0);
            myPanel->baseWidget().features(WIDGET_FEATURE_BGFIT);
            myPanel->baseWidget().setBgImage(PME_RESOURCES, "messageEnd_bg");

            // Add our buttons
            myPanel->createWidget("GameLabel", WT_LABEL, ID_GAME_LABEL);
            myPanel->getWidget(ID_GAME_LABEL)->setFont(get(RM_FONT_SCORE));
            myPanel->getWidget(ID_GAME_LABEL)->setText("Game Aborted");
            myPanel->getWidget(ID_GAME_LABEL)->setPosition(PH_CENTER, Position(PH_TOP, 20.0f));
            myPanel->getWidget(ID_GAME_LABEL)->features(WIDGET_FEATURE_BGDISABLE);

            myPanel->createWidget("GameEnterName", WT_TEXTBOX, ID_GAME_ENTERNAME);
            myPanel->getWidget(ID_GAME_ENTERNAME)->setSize(200, 48);
            myPanel->getWidget(ID_GAME_ENTERNAME)->setMargin(5, 20, 0, 20);
            myPanel->getWidget(ID_GAME_ENTERNAME)->setPosition(PH_CENTER, PH_CENTER);
            myPanel->getWidget(ID_GAME_ENTERNAME)->setFont(fntInfo);
            myPanel->getWidget(ID_GAME_ENTERNAME)->setText("PacMan");
            myPanel->getWidget(ID_GAME_ENTERNAME)->setBgSprite(PME_RESOURCES, "widget_input");

            myPanel->createWidget("GameClose", WT_BUTTON, ID_GAME_CLOSE);
            myPanel->getWidget(ID_GAME_CLOSE)->setKey(SDLK_RETURN);
            myPanel->getWidget(ID_GAME_CLOSE)->setText("");
            myPanel->getWidget(ID_GAME_CLOSE)->setBgSprite(PME_RESOURCES, "widget_close");
            myPanel->getWidget(ID_GAME_CLOSE)->setPosition(PH_CENTER, Position(PH_BOTTOM, -25.0f));
        }
    }

    // Load the music
    if(musMenu[0] <= 0 && bErrorFlag == 0)
    {
        musMenu[0] = mAudio.load(PME_RESOURCES, "menu_1", ATT_MUSIC);
        if(musMenu[0] < 0)
        {
            mC64.ILogMgr().get()->msg(LML_CRITICAL, "  [ResourceManager] Critical: could not load the menu_1 music.\n");
            bErrorFlag = 1;
        }
    }
    if(musMenu[1] <= 0 && bErrorFlag == 0)
    {
        musMenu[1] = mAudio.load(PME_RESOURCES, "menu_2", ATT_MUSIC);
        if(musMenu[1] < 0)
        {
            mC64.ILogMgr().get()->msg(LML_CRITICAL, "  [ResourceManager] Critical: could not load the menu_2 music.\n");
            bErrorFlag = 1;
        }
    }
    if(musMenu[2] <= 0 && bErrorFlag == 0)
    {
        musMenu[2] = mAudio.load(PME_RESOURCES, "menu_3", ATT_MUSIC);
        if(musMenu[2] < 0)
        {
            mC64.ILogMgr().get()->msg(LML_CRITICAL, "  [ResourceManager] Critical: could not load the menu_3 music.\n");
            bErrorFlag = 1;
        }
    }
    if(musGame[0] <= 0 && bErrorFlag == 0)
    {
        musGame[0] = mAudio.load(PME_RESOURCES, "game_1", ATT_MUSIC);
        if(musGame[0] < 0)
        {
            mC64.ILogMgr().get()->msg(LML_CRITICAL, "  [ResourceManager] Critical: could not load the game_1 music.\n");
            bErrorFlag = 1;
        }
    }
    if(musGame[1] <= 0 && bErrorFlag == 0)
    {
        musGame[1] = mAudio.load(PME_RESOURCES, "game_2", ATT_MUSIC);
        if(musGame[1] < 0)
        {
            mC64.ILogMgr().get()->msg(LML_CRITICAL, "  [ResourceManager] Critical: could not load the game_2 music.\n");
            bErrorFlag = 1;
        }
    }
    if(musGame[2] <= 0 && bErrorFlag == 0)
    {
        musGame[2] = mAudio.load(PME_RESOURCES, "game_3", ATT_MUSIC);
        if(musGame[2] < 0)
        {
            mC64.ILogMgr().get()->msg(LML_CRITICAL, "  [ResourceManager] Critical: could not load the game_3 music.\n");
            bErrorFlag = 1;
        }
    }
    if(musGame[3] <= 0 && bErrorFlag == 0)
    {
        musGame[3] = mAudio.load(PME_RESOURCES, "game_4", ATT_MUSIC);
        if(musGame[3] < 0)
        {
            mC64.ILogMgr().get()->msg(LML_CRITICAL, "  [ResourceManager] Critical: could not load the game_4 music.\n");
            bErrorFlag = 1;
        }
    }

    // Load the sounds
    if(sndThunder <= 0 && bErrorFlag == 0)
    {
        sndThunder = mAudio.load(PME_RESOURCES, "thunder", ATT_SFX);
        if(sndThunder < 0)
        {
            mC64.ILogMgr().get()->msg(LML_CRITICAL, "  [ResourceManager] Critical: could not load the thunder sound.\n");
            bErrorFlag = 1;
        }
    }
    if(sndClickOK <= 0 && bErrorFlag == 0)
    {
        sndClickOK = mAudio.load(PME_RESOURCES, "click_ok", ATT_SFX);
        if(sndClickOK < 0)
        {
            mC64.ILogMgr().get()->msg(LML_CRITICAL, "  [ResourceManager] Critical: could not load the click_ok sound.\n");
            bErrorFlag = 1;
        }
    }
    if(sndClickCancel <= 0 && bErrorFlag == 0)
    {
        sndClickCancel = mAudio.load(PME_RESOURCES, "click_cancel", ATT_SFX);
        if(sndClickCancel < 0)
        {
            mC64.ILogMgr().get()->msg(LML_CRITICAL, "  [ResourceManager] Critical: could not load the click_cancel sound.\n");
            bErrorFlag = 1;
        }
    }
    if(sndClickCancel2 <= 0 && bErrorFlag == 0)
    {
        sndClickCancel2 = mAudio.load(PME_RESOURCES, "click_cancel2", ATT_SFX);
        if(sndClickCancel2 < 0)
        {
            mC64.ILogMgr().get()->msg(LML_CRITICAL, "  [ResourceManager] Critical: could not load the click_cancel2 sound.\n");
            bErrorFlag = 1;
        }
    }
    if(sndExit <= 0 && bErrorFlag == 0)
    {
        sndExit = mAudio.load(PME_RESOURCES, "exit", ATT_SFX);
        if(sndExit < 0)
        {
            mC64.ILogMgr().get()->msg(LML_CRITICAL, "  [ResourceManager] Critical: could not load the exit sound.\n");
            bErrorFlag = 1;
        }
    }
    if(sndGameAbort <= 0 && bErrorFlag == 0)
    {
        sndGameAbort = mAudio.load(PME_RESOURCES, "game_abort", ATT_SFX);
        if(sndGameAbort < 0)
        {
            mC64.ILogMgr().get()->msg(LML_CRITICAL, "  [ResourceManager] Critical: could not load the game_abort sound.\n");
            bErrorFlag = 1;
        }
    }
    if(sndGameOver <= 0 && bErrorFlag == 0)
    {
        sndGameOver = mAudio.load(PME_RESOURCES, "game_over", ATT_SFX);
        if(sndGameOver < 0)
        {
            mC64.ILogMgr().get()->msg(LML_CRITICAL, "  [ResourceManager] Critical: could not load the game_over sound.\n");
            bErrorFlag = 1;
        }
    }
    if(sndGameStarting <= 0 && bErrorFlag == 0)
    {
        sndGameStarting = mAudio.load(PME_RESOURCES, "game_starting", ATT_SFX);
        if(sndGameStarting < 0)
        {
            mC64.ILogMgr().get()->msg(LML_CRITICAL, "  [ResourceManager] Critical: could not load the game_starting sound.\n");
            bErrorFlag = 1;
        }
    }
    if(sndGamePlayerDeath <= 0 && bErrorFlag == 0)
    {
        sndGamePlayerDeath = mAudio.load(PME_RESOURCES, "game_player_death", ATT_SFX);
        if(sndGamePlayerDeath < 0)
        {
            mC64.ILogMgr().get()->msg(LML_CRITICAL, "  [ResourceManager] Critical: could not load the game_player_death sound.\n");
            bErrorFlag = 1;
        }
    }
    if(sndGameEatPellet <= 0 && bErrorFlag == 0)
    {
        sndGameEatPellet = mAudio.load(PME_RESOURCES, "game_eat_pellet", ATT_SFX);
        if(sndGameEatPellet < 0)
        {
            mC64.ILogMgr().get()->msg(LML_CRITICAL, "  [ResourceManager] Critical: could not load the game_eat_pellet sound.\n");
            bErrorFlag = 1;
        }
    }
    if(sndGameEatPelletPower <= 0 && bErrorFlag == 0)
    {
        sndGameEatPelletPower = mAudio.load(PME_RESOURCES, "game_eat_pellet_power", ATT_SFX);
        if(sndGameEatPelletPower < 0)
        {
            mC64.ILogMgr().get()->msg(LML_CRITICAL, "  [ResourceManager] Critical: could not load the game_eat_pellet_power sound.\n");
            bErrorFlag = 1;
        }
    }
    if(sndGameEatGhost <= 0 && bErrorFlag == 0)
    {
        sndGameEatGhost = mAudio.load(PME_RESOURCES, "game_player_eat_ghost", ATT_SFX);
        if(sndGameEatGhost < 0)
        {
            mC64.ILogMgr().get()->msg(LML_CRITICAL, "  [ResourceManager] Critical: could not load the game_player_eat_ghost sound.\n");
            bErrorFlag = 1;
        }
    }

    // Load the images
    if(imgIcon <= 0 && bErrorFlag == 0) // Icon is embedded
    {
        imgIcon = mImage.load(pme_icon, pme_icon_size, "PME-icon");
        if(imgIcon < 0)
        {
            mC64.ILogMgr().get()->msg(LML_CRITICAL, "  [ResourceManager] Critical: could not create the PME Icon.\n");
            bErrorFlag = 1;
        }
    }
    if(imgMenu <= 0 && bErrorFlag == 0)
    {
        imgMenu = mImage.load(PME_RESOURCES, "menu_bg");
        if(imgMenu < 0)
        {
            mC64.ILogMgr().get()->msg(LML_CRITICAL, "  [ResourceManager] Critical: could not load the Menu BG.\n");
            bErrorFlag = 1;
        }
    }
    if(imgHallOfFame <= 0 && bErrorFlag == 0)
    {
        imgHallOfFame = mImage.load(PME_RESOURCES, "hof_bg");
        if(imgHallOfFame < 0)
        {
            mC64.ILogMgr().get()->msg(LML_CRITICAL, "  [ResourceManager] Critical: could not load the HallOfFame BG.\n");
            bErrorFlag = 1;
        }
    }

    // Load the cursor
    if(curArrow <= 0 && bErrorFlag == 0)
    {
        curArrow = mC64.ICursorMgr().load(PME_RESOURCES, "cursor");
        if(curArrow < 0)
        {
            mC64.ILogMgr().get()->msg(LML_CRITICAL, "  [ResourceManager] Critical: could not load the cursor.\n");
            bErrorFlag = 1;
        }
        else
        {
            mC64.ICursorMgr().select(curArrow);
            mC64.ICursorMgr().hide();
        }
    }

    // Load the sprites
    if(sprPacman <= 0 && bErrorFlag == 0)
    {
        sprPacman = mSprite.load(PME_RESOURCES, "pacman");
        if(sprPacman < 0)
        {
            mC64.ILogMgr().get()->msg(LML_CRITICAL, "  [ResourceManager] Critical: could not load the pacman sprite.\n");
            bErrorFlag = 1;
        }
    }
    if(sprGhostRed <= 0 && bErrorFlag == 0)
    {
        sprGhostRed = mSprite.load(PME_RESOURCES, "ghost_red");
        if(sprGhostRed < 0)
        {
            mC64.ILogMgr().get()->msg(LML_CRITICAL, "  [ResourceManager] Critical: could not load the ghost_red sprite.\n");
            bErrorFlag = 1;
        }
    }
    if(sprGhostPink <= 0 && bErrorFlag == 0)
    {
        sprGhostPink = mSprite.load(PME_RESOURCES, "ghost_pink");
        if(sprGhostPink < 0)
        {
            mC64.ILogMgr().get()->msg(LML_CRITICAL, "  [ResourceManager] Critical: could not load the ghost_pink sprite.\n");
            bErrorFlag = 1;
        }
    }
    if(sprGhostBlue <= 0 && bErrorFlag == 0)
    {
        sprGhostBlue = mSprite.load(PME_RESOURCES, "ghost_blue");
        if(sprGhostBlue < 0)
        {
            mC64.ILogMgr().get()->msg(LML_CRITICAL, "  [ResourceManager] Critical: could not load the ghost_blue sprite.\n");
            bErrorFlag = 1;
        }
    }
    if(sprGhostOrange <= 0 && bErrorFlag == 0)
    {
        sprGhostOrange = mSprite.load(PME_RESOURCES, "ghost_orange");
        if(sprGhostOrange < 0)
        {
            mC64.ILogMgr().get()->msg(LML_CRITICAL, "  [ResourceManager] Critical: could not load the ghost_orange sprite.\n");
            bErrorFlag = 1;
        }
    }
    if(sprPellet <= 0 && bErrorFlag == 0)
    {
        sprPellet = mSprite.load(PME_RESOURCES, "pellet");
        if(sprPellet < 0)
        {
            mC64.ILogMgr().get()->msg(LML_CRITICAL, "  [ResourceManager] Critical: could not load the pellet sprite.\n");
            bErrorFlag = 1;
        }
    }
    if(sprPelletPower <= 0 && bErrorFlag == 0)
    {
        sprPelletPower = mSprite.load(PME_RESOURCES, "pellet_power");
        if(sprPelletPower < 0)
        {
            mC64.ILogMgr().get()->msg(LML_CRITICAL, "  [ResourceManager] Critical: could not load the pellet_power sprite.\n");
            bErrorFlag = 1;
        }
    }
    if(sprTargets <= 0 && bErrorFlag == 0)
    {
        sprTargets = mSprite.load(PME_RESOURCES, "targets");
        if(sprTargets < 0)
        {
            mC64.ILogMgr().get()->msg(LML_CRITICAL, "  [ResourceManager] Critical: could not load the targets sprite.\n");
            bErrorFlag = 1;
        }
    }
    
    // Init our random seed. It is a good place as all the loading stuff will make it more "random"
    auto hitime = std::chrono::high_resolution_clock::now().time_since_epoch().count();
    Main::Instance().ITool().randSeedMWC((Sint32)hitime);
    Main::Instance().ITool().randSeedWELL((Sint32)hitime);

    // Check for any error
    if(bErrorFlag != 0)
    {
        free();
        return PME_BREAK;
    }
    
    return 0;
}

// Free all resources.
Sint32 ResourceManager::free()
{
    Main& mC64 = Main::Instance();

    if(sprPacman > 0) mC64.ISpriteMgr().close(sprPacman);
    sprPacman = 0;
    if(sprGhostRed > 0) mC64.ISpriteMgr().close(sprGhostRed);
    sprGhostRed = 0;
    if(sprGhostPink > 0) mC64.ISpriteMgr().close(sprGhostPink);
    sprGhostPink = 0;
    if(sprGhostBlue > 0) mC64.ISpriteMgr().close(sprGhostBlue);
    sprGhostBlue = 0;
    if(sprGhostOrange > 0) mC64.ISpriteMgr().close(sprGhostOrange);
    sprGhostOrange = 0;
    if(sprPellet > 0) mC64.ISpriteMgr().close(sprPellet);
    sprPellet = 0;
    if(sprPelletPower > 0) mC64.ISpriteMgr().close(sprPelletPower);
    sprPelletPower = 0;
    if(sprTargets > 0) mC64.ISpriteMgr().close(sprTargets);
    sprTargets = 0;

    if(curArrow > 0) mC64.ICursorMgr().close(curArrow);
    curArrow = 0;
    
    if(imgIcon > 0) mC64.IImageMgr().close(imgIcon);
    imgIcon = 0;
    if(imgMenu > 0) mC64.IImageMgr().close(imgMenu);
    imgMenu = 0;
    if(imgHallOfFame > 0) mC64.IImageMgr().close(imgHallOfFame);
    imgHallOfFame = 0;
   
    if(sndThunder > 0) mC64.IAudioTrackMgr().close(sndThunder);
    sndThunder = 0;
    if(sndClickOK > 0) mC64.IAudioTrackMgr().close(sndClickOK);
    sndClickOK = 0;
    if(sndClickCancel2 > 0) mC64.IAudioTrackMgr().close(sndClickCancel2);
    sndClickCancel2 = 0;
    if(sndClickCancel > 0) mC64.IAudioTrackMgr().close(sndClickCancel);
    sndClickCancel = 0;
    if(sndExit > 0) mC64.IAudioTrackMgr().close(sndExit);
    sndExit = 0;
    if(sndGameAbort > 0) mC64.IAudioTrackMgr().close(sndGameAbort);
    sndGameAbort = 0;
    if(sndGameOver > 0) mC64.IAudioTrackMgr().close(sndGameOver);
    sndGameOver = 0;
    if(sndGameStarting > 0) mC64.IAudioTrackMgr().close(sndGameStarting);
    sndGameStarting = 0;
    if(sndGamePlayerDeath > 0) mC64.IAudioTrackMgr().close(sndGamePlayerDeath);
    sndGamePlayerDeath = 0;
    if(sndGameEatPellet > 0) mC64.IAudioTrackMgr().close(sndGameEatPellet);
    sndGameEatPellet = 0;
    if(sndGameEatPelletPower > 0) mC64.IAudioTrackMgr().close(sndGameEatPelletPower);
    sndGameEatPelletPower = 0;
    if(sndGameEatGhost > 0) mC64.IAudioTrackMgr().close(sndGameEatGhost);
    sndGameEatGhost = 0;

    if(musMenu[0] > 0) mC64.IAudioTrackMgr().close(musMenu[0]);
    musMenu[0] = 0;
    if(musMenu[1] > 0) mC64.IAudioTrackMgr().close(musMenu[1]);
    musMenu[1] = 0;
    if(musMenu[2] > 0) mC64.IAudioTrackMgr().close(musMenu[2]);
    musMenu[2] = 0;
    if(musGame[0] > 0) mC64.IAudioTrackMgr().close(musGame[0]);
    musGame[0] = 0;
    if(musGame[1] > 0) mC64.IAudioTrackMgr().close(musGame[1]);
    musGame[1] = 0;
    if(musGame[2] > 0) mC64.IAudioTrackMgr().close(musGame[2]);
    musGame[2] = 0;
    if(musGame[3] > 0) mC64.IAudioTrackMgr().close(musGame[3]);
    musGame[3] = 0;
    
    if(fntConsole > 0) mC64.IFontMgr().close(fntConsole);
    fntConsole = 0;
    if(fntInfo > 0) mC64.IFontMgr().close(fntInfo);
    fntInfo = 0;
    if(fntScore > 0) mC64.IFontMgr().close(fntScore);
    fntScore = 0;
    
    if(guiMenu > 0) mC64.IGUIMgr().close(guiMenu);
    guiMenu = 0;

    if(guiGame > 0) mC64.IGUIMgr().close(guiGame);
    guiGame = 0;
    return 0;
}

// Get resources which are created with load()
Sint32 ResourceManager::get(Sint32 iID) const
{
    SpriteMgr& mSprite = Main::Instance().ISpriteMgr();
    Sint32 iTmp;

    switch(iID)
    {
    case RM_PANEL_MENU: return guiMenu;
    case RM_PANEL_GAME: return guiGame;

    case RM_FONT_CONSOLE: return fntConsole;
    case RM_FONT_INFO: return fntInfo;
    case RM_FONT_SCORE: return fntScore;

    case RM_MUS_MENU: return musMenu[Main::Instance().ITool().randMWC() % 3];
    case RM_MUS_GAME: return musGame[Main::Instance().ITool().randMWC() % 4];

    case RM_SND_THUNDER: return sndThunder;
    case RM_SND_CLICKOK: return sndClickOK;
    case RM_SND_CLICKCANCEL: return sndClickCancel;
    case RM_SND_CLICKCANCEL2: return sndClickCancel2;
    case RM_SND_EXIT: return sndExit;
    case RM_SND_GAMEABORT: return sndGameAbort;
    case RM_SND_GAMEOVER: return sndGameOver;
    case RM_SND_GAMESTARTING: return sndGameStarting;
    case RM_SND_GAMEPLAYERDEATH: return sndGamePlayerDeath;
    case RM_SND_GAMEEATPELLET: return sndGameEatPellet;
    case RM_SND_GAMEEATPELLETPOWER: return sndGameEatPelletPower;
    case RM_SND_GAMEEATGHOST: return sndGameEatGhost;

    case RM_IMG_ICON: return imgIcon;
    case RM_IMG_MENU: return imgMenu;
    case RM_IMG_HOF: return imgHallOfFame;

    case RM_SPR_PACMAN:
    {
        iTmp = mSprite.child(sprPacman);
        Sprite* pSpr = mSprite.get(sprPacman);
        if(bPacManDeathAnim == 0) pSpr->setAnimRange(SPR_STATE_NORMAL, 10, 12); // Disable PacMan death animation
        else pSpr->setAnimRange(SPR_STATE_NORMAL, 0, 12); // Enable PacMan death animation
        return iTmp;
    }
    case RM_SPR_GHOSTRED: return mSprite.child(sprGhostRed);
    case RM_SPR_GHOSTPINK: return mSprite.child(sprGhostPink);
    case RM_SPR_GHOSTBLUE: return mSprite.child(sprGhostBlue);
    case RM_SPR_GHOSTORANGE: return mSprite.child(sprGhostOrange);
    case RM_SPR_PELLET: return sprPellet;
    case RM_SPR_PELLETPOWER: return mSprite.child(sprPelletPower);
    case RM_SPR_TARGETS: return mSprite.child(sprTargets);

    default: return PME_BREAK;
    }
}

Sint32 ResourceManager::setPacManDeathAnim(Sint32 bFlag)
{
    if(bFlag <= 0) bPacManDeathAnim = 0;
    else bPacManDeathAnim = 1;
    return 0;
}
