/*----------------------------------------------------------------------
Pac-Man Evolution - Roberto Prieto
 Copyright (C) 2018-2025 MegaStorm Systems
contact@megastormsystems.com - http://www.megastormsystems.com

This software is provided 'as-is', without any express or implied
warranty.  In no event will the authors be held liable for any damages
arising from the use of this software.

Permission is granted to anyone to use this software for any purpose,
including commercial applications, and to alter it and redistribute it
freely, subject to the following restrictions:

1. The origin of this software must not be misrepresented; you must not
claim that you wrote the original software. If you use this software
in a product, an acknowledgment in the product documentation would be
appreciated but is not required.
2. Altered source versions must be plainly marked as such, and must not be
misrepresented as being the original software.
3. This notice may not be removed or altered from any source distribution.

------------------------------------------------------------------------

Pac-Man Evolution

------------------------------------------------------------------------ */

// Includes
#include "Pac-Man_Evolution.h"
#include "Menu.h"
#include "HoF.h"
#include "ResourceManager.h"
#include "GameField.h"
#include "BrainsFactory.h"
#include "EVNTrainer.h"

// Global vars and our render wrapper callback
GlobalStatus globalStatus;
Menu* pMenu = nullptr;
HoF* pHoF = nullptr;
GameField* pGameField = nullptr;

Sint32 renderWrapper(Sint32 iMode, void* pData)
{
    Sint32 iRet = 0;

    switch(globalStatus.iRenderScreen)
    {
    case PME_SCREEN_MENU:
        if(pMenu != nullptr) iRet = pMenu->render(iMode);
        break;
    case PME_SCREEN_HOF:
        if(pHoF != nullptr) iRet = pHoF->render(iMode);
        break;
    case PME_SCREEN_GAME:
        if(pGameField != nullptr) iRet = pGameField->render(iMode);
        break;
    }    
    return iRet;
}
#include "ArtificialNeuralNet.h"

// Main body
int main(int argc, char* argv[])
{    
    Sint32 iDone = PME_ENTRY, i;
    SDL_Event eEvent;
    Screen* pScreen;

    // Initialize CMem
    CMem::setLogLevel(CMem::eMemStatsLevel::MSL_NORMAL);
    atexit(CMem::destroy);

    // Initialize CRM64Pro    
    Main& mC64 = Main::Instance();
    Log& mLog = *mC64.ILogMgr().get();
    mLog.init(GAME_VERSION, LL_HIGH, LM_FILE | LM_STDOUT, "Pac-Man_Evolution.log");
    mC64.ITimer().init();
    mC64.ITimer().setRate(60, PME_LOGIC_RATE);
    mC64.IConfigMgr().iMTFriendly = 1;
    
    // Initialize the screen
    pScreen = mC64.IConfigMgr().get();
    pScreen->setDriver(CRD_OPENGL);
    pScreen->setSize(1024, 768);
    //pScreen->setMode(CSM_FULLSCREEN);
    pScreen->setTitle(GAME_VERSION);
    if(pScreen->show() < 0)
    {
        Main::Terminate();
        return -1;
    }
    pScreen->setRenderCallback(renderWrapper); 
    mC64.IConfigMgr().audioInit(AF_HIGH, AS_16, AM_STEREO);    
    
    // Create the resource manager and load all resources
    if(ResourceManager::Instance().load() != 0) 
    {
        mC64.IConfigMgr().audioClose();
        ResourceManager::Terminate();
        Main::Terminate();
        return -2;
    }    
    pScreen->setIcon(mC64.IImageMgr().get(ResourceManager::Instance().get(RM_IMG_ICON))->getSurface());
    
    // Create the menu, hof and gamefield systems
    pMenu = new(std::nothrow) Menu();
    pHoF = new(std::nothrow) HoF();
    pGameField = new(std::nothrow) GameField(&globalStatus);

    // MegaStorm and CRM64Pro introduction
    mC64.intro();

    // --- Quickmode: Training game ---
    /*globalStatus.workBench.iTraining = 1;
    globalStatus.workBench.iExecutions = 100;
    globalStatus.workBench.iSpeed = 40;
    globalStatus.workBench.iTime = 5;
    globalStatus.workBench.iPacManBrain = PME_BRAIN_TYPE_FIXED; // PME_BRAIN_TYPE_HUMAN;
    globalStatus.workBench.iGhostRedBrain = PME_BRAIN_TYPE_TRAINING0; // PME_BRAIN_TYPE_EVOLVED; //PME_BRAIN_TYPE_TRAINING0;
    globalStatus.workBench.iGhostPinkBrain = PME_BRAIN_TYPE_TRAINING0;
    globalStatus.workBench.iGhostBlueBrain = PME_BRAIN_TYPE_TRAINING0;
    globalStatus.workBench.iGhostOrangeBrain = PME_BRAIN_TYPE_TRAINING0;
    strcpy(globalStatus.workBench.szOutputCSV, "Fitness.csv");
    // Setup new speed
    mC64.ITimer().setRate(60, PME_LOGIC_RATE * globalStatus.workBench.iSpeed);
    // Take care of the training
    EVNTrainer::Instance().execute(globalStatus, *pGameField);
    // Restore default speed
    mC64.ITimer().setRate(60, PME_LOGIC_RATE);
    iDone = PME_ENTRY;
    // ---------------------------------

    /**/
    // Main loop
    while(iDone != PME_EXIT)
    {
        // 1.First execution or after a game match
        if(iDone == PME_ENTRY)
        {
            mC64.IAudioTrackMgr().get(ResourceManager::Instance().get(RM_MUS_MENU))->play(-1);
            pScreen->fadeToImage(ResourceManager::Instance().get(RM_IMG_MENU), 500);
            mC64.ICursorMgr().show();            
            mC64.IGUIMgr().getPanel(ResourceManager::Instance().get(RM_PANEL_MENU))->baseWidget().show();
            globalStatus.clear();
            globalStatus.iRenderScreen = PME_SCREEN_MENU;
            iDone = PME_LOOP;
        }

        // 2.Standard/Evolution games
        else if((iDone == PME_GAME_STANDARD) || (iDone == PME_GAME_EVOLUTION))
        {    
            // Hide cursor and GUI as they are not used in the game
            mC64.ICursorMgr().hide();
            mC64.IGUIMgr().getPanel(ResourceManager::Instance().get(RM_PANEL_MENU))->baseWidget().hide();

            globalStatus.iHighestScore = pHoF->getHighest(iDone);
            globalStatus.iLowestScore = pHoF->getLowest(iDone);
            globalStatus.iGameType = iDone;
            iDone = pGameField->init();
            while(iDone == PME_LOOP)
            {
                iDone = pGameField->execute();
                if(iDone == PME_MAZE_END) iDone = pGameField->nextMaze();
            }
            pGameField->close();
            pHoF->store(globalStatus.iGameType, globalStatus.iPoints, globalStatus.szName);
            iDone = PME_ENTRY;
        }

        // 3.Workbench games
        else if(iDone == PME_GAME_WORKBENCH)
        {
            // Hide cursor and GUI as they are not used in the game
            mC64.ICursorMgr().hide();
            mC64.IGUIMgr().getPanel(ResourceManager::Instance().get(RM_PANEL_MENU))->baseWidget().hide();

            // todo meter aqui el panel de workbench            
            globalStatus.workBench.iExecutions = 1;
            globalStatus.workBench.iSpeed = 3;
            globalStatus.workBench.iTime = 0;
            globalStatus.workBench.iPacManBrain = PME_BRAIN_TYPE_FIXED; //PME_BRAIN_TYPE_HUMAN;// PME_BRAIN_TYPE_FIXED;
            globalStatus.workBench.iGhostRedBrain = PME_BRAIN_TYPE_EVOLVED; //PME_BRAIN_TYPE_EVOLVED;
            globalStatus.workBench.iGhostPinkBrain = PME_BRAIN_TYPE_EVOLVED;
            globalStatus.workBench.iGhostBlueBrain = PME_BRAIN_TYPE_EVOLVED;
            globalStatus.workBench.iGhostOrangeBrain = PME_BRAIN_TYPE_EVOLVED;
            //strcpy(globalStatus.workBench.szOutputCSV, "workbench.csv");
            // ---------------

            // Setup new speed
            mC64.ITimer().setRate(60, PME_LOGIC_RATE * globalStatus.workBench.iSpeed);
            // Open output CSV
            FILE* fp = fopen(globalStatus.workBench.szOutputCSV, "wt");
            i = 0;
            while(i < globalStatus.workBench.iExecutions)
            {
                globalStatus.iHighestScore = pHoF->getHighest(iDone);
                globalStatus.iLowestScore = pHoF->getLowest(iDone);
                globalStatus.iGameType = PME_GAME_WORKBENCH;
                iDone = pGameField->init();
                while(iDone == PME_LOOP)
                {
                    iDone = pGameField->execute();
                    if(iDone == PME_MAZE_END) iDone = pGameField->nextMaze();
                }
                pGameField->close();
                // Write points to the CSV after finishing a match
                fprintf(fp, "%d,", globalStatus.iPoints);
                fflush(fp);                
                ++i;
                // Output fitness
                //mLog.msg(LML_INFO, "  [Pac-Man Evolution] Info: Last game fitness %.6f.\n", 1.0/(double)(globalStatus.iPoints));
                globalStatus.iPoints = 0;
            }            
            // Restore default speed and close the CSV file
            fclose(fp);
            mC64.ITimer().setRate(60, PME_LOGIC_RATE);
            iDone = PME_ENTRY;
        }
        
        // 4.Manage Hall of Fame
        else if(iDone == PME_HOF)
        {
            pScreen->fadeToImage(ResourceManager::Instance().get(RM_IMG_HOF), 250);
            mC64.IGUIMgr().getPanel(ResourceManager::Instance().get(RM_PANEL_MENU))->baseWidget().hide();
            globalStatus.iRenderScreen = PME_SCREEN_HOF;
            iDone = PME_LOOP;
        }

        // 5.Events loop
        while(mC64.update(&eEvent))
        {
            switch(eEvent.type)
            {
                // Quit signal
            case SDL_EVENT_QUIT:                
                iDone = PME_EXIT;
                break;
                // Any pressed key or mouse click return from HoF to Menu
            case SDL_EVENT_KEY_DOWN:
            case SDL_EVENT_MOUSE_BUTTON_DOWN:
                if(globalStatus.iRenderScreen == PME_SCREEN_HOF)
                {
                    pScreen->fadeToImage(ResourceManager::Instance().get(RM_IMG_MENU), 250);
                    mC64.IGUIMgr().getPanel(ResourceManager::Instance().get(RM_PANEL_MENU))->baseWidget().show();
                    globalStatus.iRenderScreen = PME_SCREEN_MENU;
                }
                break;
                // Handle C64 events
            case C64_EVENT:
                // Process the events in the Menu
                if(eEvent.user.code == C64_EVENT_WIDGET) iDone = pMenu->processEvent(*static_cast<Sint32*>(eEvent.user.data1), *static_cast<Sint32*>(eEvent.user.data2));
                break;
            }
        }
    }
    
    // Close and terminate   
    mC64.IAudioTrackMgr().get(ResourceManager::Instance().get(RM_SND_EXIT))->play();
    mC64.IAudioTrackMgr().fadeOutTag(ATT_MUSIC, 1000);
    pScreen->fadeToColor(0, 0, 0, 1000);        
    delete pGameField;
    delete pHoF;
    delete pMenu;    
    BrainsFactory::Terminate();
    ResourceManager::Terminate();
    EVNTrainer::Terminate();
    mLog.msg(LML_INFO, "\n [Pac-Man Evolution] Info: game closed.\n");
    mC64.IConfigMgr().audioClose();
    Main::Terminate();      
    return 0;
}

// GlobalStatus
GlobalStatus::GlobalStatus()
{
    clear();
}

GlobalStatus::~GlobalStatus()
{
}

Sint32 GlobalStatus::clear()
{
	iRenderScreen = -1;
	strcpy(szName, "PacMan");
	iPoints = 0;
    iHighestScore = iLowestScore = 0;
    iGameType = -1;
    workBench.iTraining = 0;
    workBench.iExecutions = 100;
    workBench.iSpeed = 20;
    workBench.iTime = 0;
    strcpy(workBench.szOutputCSV, "workbench.csv");
    workBench.iPacManBrain = workBench.iGhostRedBrain = workBench.iGhostPinkBrain = PME_OBJECT_NULL;
    workBench.iGhostBlueBrain = workBench.iGhostOrangeBrain = PME_OBJECT_NULL;
    return 0;
}
