/*----------------------------------------------------------------------
Pac-Man Evolution - Roberto Prieto
 Copyright (C) 2018-2025 MegaStorm Systems
contact@megastormsystems.com - http://www.megastormsystems.com

This software is provided 'as-is', without any express or implied
warranty.  In no event will the authors be held liable for any damages
arising from the use of this software.

Permission is granted to anyone to use this software for any purpose,
including commercial applications, and to alter it and redistribute it
freely, subject to the following restrictions:

1. The origin of this software must not be misrepresented; you must not
claim that you wrote the original software. If you use this software
in a product, an acknowledgment in the product documentation would be
appreciated but is not required.
2. Altered source versions must be plainly marked as such, and must not be
misrepresented as being the original software.
3. This notice may not be removed or altered from any source distribution.

------------------------------------------------------------------------

PacMan class

------------------------------------------------------------------------ */

#include "ObjectsPacMan.h"
#include "GameField.h"
#include "ResourceManager.h"
#include "BrainsFactory.h"

// PacMan constructor.
PacMan::PacMan(Sint32 iMX, Sint32 iMY, GameField* GF) : Actor(PME_OBJECT_PACMAN, iMX, iMY, GF)
{
    // Init vars    
    iLifes = PME_PACMAN_START_LIFES;
    iSpeed = PME_PACMAN_START_SPEED;
    iPointsForEatingGhostsPerPelletPower = 0;

    // Create our state objects
    pStateInit = new(std::nothrow) PacManStateInit("Init");
    pStateEvading = new(std::nothrow) PacManStateEvading("Evading");
    pStateChasing = new(std::nothrow) PacManStateChasing("Chasing");
    pStateDeath = new(std::nothrow) PacManStateDeath("Death");

    #ifdef DEBUG_INTERNAL
        Main::Instance().ILogMgr().get()->msg(LML_LOW, "     [PacMan] PacMan ID '%d' created.\n", iID);
    #endif
}

// PacMan destructor.
PacMan::~PacMan()
{
    iLifes = 0;
    delete pStateInit;
    pStateInit = nullptr;
    delete pStateEvading;
    pStateEvading = nullptr;
    delete pStateChasing;
    pStateChasing = nullptr;
    delete pStateDeath;
    pStateDeath = nullptr;
    #ifdef DEBUG_INTERNAL
        Main::Instance().ILogMgr().get()->msg(LML_LOW, "     [PacMan] PacMan ID '%d' deleted.\n", iID);
    #endif
}

// PacMan logic execution.
// Return PME_ACTOR_ALIVE, PME_ACTOR_SPECIAL or PME_GAME_OVER
Sint32 PacMan::execute()
{
    Sint32 iRet = PME_ACTOR_ALIVE;    
    
    // With no more lifes left...
    if(iLifes <= 0) return PME_GAME_OVER;

    // Execute current state.
    // Note the states have full access to all our methods/attributes
    if(pCurrentState) pCurrentState->execute(this);
    
    // Call base method (set the object sprite position)
    Actor::execute();

    // Check for init, chase and death states -> put us in special mode: no wavetick counter!
    if(stateCurrentCheck(*pStateInit) || stateCurrentCheck(*pStateChasing) || stateCurrentCheck(*pStateDeath)) iRet = PME_ACTOR_SPECIAL;

    // Return
    return iRet;
}

// Debug PacMan object.
Sint32 PacMan::debug(Sint32 iMode)
{
    Main &mC64 = Main::Instance();
    string sDebug, sTmp;
    Font* pFont = Main::Instance().IFontMgr().get(ResourceManager::Instance().get(RM_FONT_CONSOLE));

    // Call our base class (render target sprite)
    Actor::debug(iMode);

    // PacMan information
    pFont->setPosition(PME_DEBUG_PANEL_PACMAN_X, static_cast<float>(PME_DEBUG_PANEL_PACMAN_Y + iMazeRenderingY));
    pFont->render("PacMan info");

    // Maze and pixel position
    mC64.ITool().intToStrDec(iMazeX, sTmp);
    sDebug = "Position: [";
    sDebug += sTmp;
    sDebug += ",";
    mC64.ITool().intToStrDec(iMazeY, sTmp);
    sDebug += sTmp;
    sDebug += "] (";
    mC64.ITool().intToStrDec(iPositionX, sTmp);
    sDebug += sTmp;
    sDebug += ",";
    mC64.ITool().intToStrDec(iPositionY, sTmp);
    sDebug += sTmp;
    sDebug += ")";
    pFont->setPosition(PME_DEBUG_PANEL_PACMAN_X, static_cast<float>(PME_DEBUG_PANEL_PACMAN_Y + iMazeRenderingY + 16));
    pFont->render(sDebug);
    
    // Speed
    sDebug = "Speed: ";
    mC64.ITool().intToStrDec(iSpeed, sTmp);
    sDebug += sTmp;
    pFont->setPosition(PME_DEBUG_PANEL_PACMAN_X, static_cast<float>(PME_DEBUG_PANEL_PACMAN_Y + iMazeRenderingY + 32));
    pFont->render(sDebug);

    // Target        
    mC64.ITool().intToStrDec(pointTarget.iX, sTmp);
    sDebug = "Target: [";
    sDebug += sTmp;
    sDebug += ",";
    mC64.ITool().intToStrDec(pointTarget.iY, sTmp);
    sDebug += sTmp;
    sDebug += "]";
    pFont->setPosition(PME_DEBUG_PANEL_PACMAN_X, static_cast<float>(PME_DEBUG_PANEL_PACMAN_Y + iMazeRenderingY + 48));
    pFont->render(sDebug);

    // State
    pCurrentState->getName(sDebug);
    sDebug = "State: " + sDebug;
    pFont->setPosition(PME_DEBUG_PANEL_PACMAN_X, static_cast<float>(PME_DEBUG_PANEL_PACMAN_Y + iMazeRenderingY + 64));
    pFont->render(sDebug);

    // Brain name
    if(BrainsFactory::Instance().getName(idBrain, sDebug) == -1) sDebug = "Empty";
    sDebug = "Brain: " + sDebug;
    pFont->setPosition(PME_DEBUG_PANEL_PACMAN_X, static_cast<float>(PME_DEBUG_PANEL_PACMAN_Y + iMazeRenderingY + 80));
    pFont->render(sDebug);

    return 0;
}

// Return number of lifes.
Sint32 PacMan::getLifes()
{
    return iLifes;
}

// Message for going to Init state.
Sint32 PacMan::msgGoInit()
{
    // Call base method
    Actor::msgGoInit(); 
    stateChange(pStateInit);
    return 0;
}

// PacMan reacts changing to chasing state.
Sint32 PacMan::msgPelletPowerEaten(Sint32 iX, Sint32 iY)
{
    stateChange(pStateChasing);
    return 0;
}

// PacMan reaction to a collision event.
// return 0: eaten by a ghost. Going death state.
// return >0: points for eating a ghost.
Sint32 PacMan::msgGhostCollision()
{
    // If I am on chasing state... then a ghost was eaten :)
    if(stateCurrentCheck(*pStateChasing))
    {
        if(iPointsForEatingGhostsPerPelletPower == 0) iPointsForEatingGhostsPerPelletPower = PME_POINTS_EAT_GHOST;
        else iPointsForEatingGhostsPerPelletPower *= 2;
        return iPointsForEatingGhostsPerPelletPower;        
    }
    // Otherwise... we were eating by a ghost
    else stateChange(pStateDeath);
    return 0;
}

// Given a desired target (could be valid or not), we try to reach it applying our rules:
// - try to reach target using euclidean distance
// - increase the cost of going back, it is allowed but with a penalty for avoiding loops
// - keep moving
// - result is stored on vPath
Sint32 PacMan::applyMovementRules(Sint32 iTX, Sint32 iTY)
{
    Sint32 i, iSelected = 0;
    double dLowest = 1000.0f;
    vector<MazePoint> vMovementOptions;

    // 1.Get possible movement options. ToDO: tunnels?
    pGameField->getMovementOptions(iMazeX, iMazeY, vMovementOptions);

    // 2.1.If there is only a possible way, avoid more calcs
    if(vMovementOptions.size() == 1) iSelected = 0;

    // 2.2.Assign distances to target and select the closest one.
    // In case of match, we select the first one: it is already prioritized in getMovementOptions()
    else
    {
        for(i = 0; i < vMovementOptions.size(); ++i) euclideanDistance(iTX, iTY, vMovementOptions[i]);

        // Increase the cost(distance) of going back
        for(i = 0; i < vMovementOptions.size(); ++i)
        {
            if((vMovementOptions[i].iX == iMazePrevX) && (vMovementOptions[i].iY == iMazePrevY))
            {
                // 1.75 is the a very good value: avoid to fall in loops and turn when in front of a ghost
                // Smaller than that and it will loop on corners more frequently and greater than that and will not evade ghost
                vMovementOptions[i].dDistance += 1.75; 
            }
        }

        // Get the lowest distance
        for(i = 0; i < vMovementOptions.size(); ++i)
        {
            if(dLowest > vMovementOptions[i].dDistance)
            {
                dLowest = vMovementOptions[i].dDistance;
                iSelected = i;
            }
        }
    }

    // 4.Add the selected point to the path
    vPath.push_back(vMovementOptions[iSelected]);
    return 0;
}

// Init State
// Show the Get Ready message waiting PME_GETREADY_TIME before going to Evading state
PacManStateInit::PacManStateInit(const string& sN) : State(sN) { iTime = 0; }
void PacManStateInit::enter(Actor* pAct)
{
    PacMan* pPM = reinterpret_cast<PacMan*>(pAct);

    // Call our base class method
    State::enter(pAct);
    
    // Reset our movemements vars    
    pPM->eAM = Actor::eActorMoving::AM_NEXT_STEP;
    pPM->vPath.clear();

    iTime = Main::Instance().ITimer().getTicksNow();
}
void PacManStateInit::execute(Actor* pAct)
{
    PacMan* pPM = reinterpret_cast<PacMan*>(pAct);
    if(Main::Instance().ITimer().getTicksNow() > (iTime + pPM->pGameField->messageGetReady(-1))) pPM->stateChange(pPM->pStateEvading);
    else pPM->pGameField->messageGetReady(1);    
}
void PacManStateInit::exit(Actor* pAct)
{
    // Call our base class method
    State::exit(pAct);

    reinterpret_cast<PacMan*>(pAct)->pGameField->messageGetReady(0);
}

// Evading State
// Only use execute() method, the rest of methods, we use our base class
PacManStateEvading::PacManStateEvading(const string& sN) : State(sN) { iTX = iTY = iITX = iITY = 0; bGetNewTarget = true; }
void PacManStateEvading::execute(Actor* pAct)
{
    Sint32 iTargetType;
    MazePoint pointImmediateTarget;
    PacMan* pPM = reinterpret_cast<PacMan*>(pAct);

    // 1.When we reach our target, request to get a new target
    if((pPM->iMazeX == iTX) && (pPM->iMazeY == iTY)) bGetNewTarget = true;

    // 2.At each new maze point, request once to get a new target
    if((pPM->iMazeX != iITX) || (pPM->iMazeY != iITY))
    {
        iITX = pPM->iMazeX;
        iITY = pPM->iMazeY;
        bGetNewTarget = true;
    }

    // 3.If our brain has an "inmmediate target" type, request new target
    iTargetType = BrainsFactory::Instance().getTargetType(pPM->idBrain);
    if(iTargetType == PME_BRAIN_IMMEDIATE_TARGET) bGetNewTarget = true;
    
    // 4.Call to the ghost brain for getting a new target in (iTX,iTY)
    if(bGetNewTarget)
    {
        pPM->think(iTX, iTY);
        pPM->pointTarget.iX = iTX;
        pPM->pointTarget.iY = iTY;
        bGetNewTarget = false;
    }

    // 5.1.With "high-level target", apply PacMan movement rules
    if(iTargetType == PME_BRAIN_HIGHLEVEL_TARGET)
    {
        if(pPM->vPath.size() == 0)
        {
            pPM->applyMovementRules(iTX, iTY);
        }
    }
    // 5.2.With an "immediate target" type just go to the new requested target
    else
    {
        pPM->vPath.clear();
        pointImmediateTarget.iX = iTX;
        pointImmediateTarget.iY = iTY;
        pPM->vPath.push_back(pointImmediateTarget);
    }
}

// Chasing State
// execute() method is the same of evading state.
PacManStateChasing::PacManStateChasing(const string& sN) : State(sN) { iTicks = 0; }
void PacManStateChasing::enter(Actor* pAct)
{
    PacMan* pPM = reinterpret_cast<PacMan*>(pAct);

    // Call our base class method
    State::enter(pAct);

    // Reset the points for eating successive ghosts during the same empowered time
    pPM->iPointsForEatingGhostsPerPelletPower = 0;

    // Init our ticks based on the baseline LFR rate(40).
    iTicks = static_cast<Sint32>((PME_PACMAN_ENPOWERED_TIME * 40) / 1000);

    // Change the GlobalWaveMode to evading
    pPM->pGameField->setWaveMode(PME_GLOBAL_WAVE_EVADING);
}
void PacManStateChasing::execute(Actor* pAct)
{
    PacMan* pPM = reinterpret_cast<PacMan*>(pAct);
    
    // Control the time spent on this mode and update the GlobalWaveMode
    if(iTicks < 0) pPM->stateChange(pPM->pStateEvading);
    else
    {
        // When 1/4 of ticks left, change GlobalWaveMode
        if(iTicks < ((PME_PACMAN_ENPOWERED_TIME * 40) / 4) / 1000) pPM->pGameField->setWaveMode(PME_GLOBAL_WAVE_EVADING_END);
        pPM->pStateEvading->execute(pAct);
    }
    --iTicks;
}
void PacManStateChasing::exit(Actor* pAct)
{
    // Call our base class
    State::exit(pAct);

    // Restore previous GlobalWaveMode
    reinterpret_cast<PacMan*>(pAct)->pGameField->restoreWaveMode();
}

// Death State
// - play death anim and once finished, substract a life
// - call GameField initObjects()
PacManStateDeath::PacManStateDeath(const string& sN) : State(sN) {}
void PacManStateDeath::enter(Actor* pAct)
{
    // Call our base class method
    State::enter(pAct);

    PacMan* pPM = reinterpret_cast<PacMan*>(pAct);
    Sprite* sprObj = Main::Instance().ISpriteMgr().get(pPM->sprID);
    
    // Set death sprite
    pPM->eSS = Actor::eSpriteState::SS_DEATH;
    sprObj->selectAnim(SPR_STATE_NORMAL);     
}
void PacManStateDeath::execute(Actor* pAct)
{
    PacMan* pPM = reinterpret_cast<PacMan*>(pAct);
    Sprite* sprObj = Main::Instance().ISpriteMgr().get(pPM->sprID);

    // Wait till the animation is over
    if(sprObj->status() == C64_STATUS_END)
    {
        --pPM->iLifes;
        pPM->pGameField->initObjects(); // Re-init all objects changing to Init state
    }    
}