/*----------------------------------------------------------------------
Pac-Man Evolution - Roberto Prieto
 Copyright (C) 2018-2025 MegaStorm Systems
contact@megastormsystems.com - http://www.megastormsystems.com

This software is provided 'as-is', without any express or implied
warranty.  In no event will the authors be held liable for any damages
arising from the use of this software.

Permission is granted to anyone to use this software for any purpose,
including commercial applications, and to alter it and redistribute it
freely, subject to the following restrictions:

1. The origin of this software must not be misrepresented; you must not
claim that you wrote the original software. If you use this software
in a product, an acknowledgment in the product documentation would be
appreciated but is not required.
2. Altered source versions must be plainly marked as such, and must not be
misrepresented as being the original software.
3. This notice may not be removed or altered from any source distribution.

------------------------------------------------------------------------

Ghost class

------------------------------------------------------------------------ */

#include "ObjectsGhost.h"
#include "GameField.h"
#include "ResourceManager.h"
#include "MapSearchAStar.h"
#include "BrainsFactory.h"

// Ghost constructor.
Ghost::Ghost(Sint32 iObjID, Sint32 iMX, Sint32 iMY, GameField* GF) : Actor(iObjID, iMX, iMY, GF)
{
    // Custom attributes for each ghost
    if(iObjID == PME_OBJECT_GHOST_RED)
    {
        iSpeed = PME_GHOST_START_SPEED;
        iGoingOutTrigger = 0;
        pointScatteringTarget.iX = 25;
        pointScatteringTarget.iY = -3;
    }
    else if(iObjID == PME_OBJECT_GHOST_PINK)
    {
        iSpeed = PME_GHOST_START_SPEED;
        iGoingOutTrigger = 10;
        pointScatteringTarget.iX = 2;
        pointScatteringTarget.iY = -3;        
    }
    else if(iObjID == PME_OBJECT_GHOST_BLUE)
    {
        iSpeed = PME_GHOST_START_SPEED;
        iGoingOutTrigger = 20;
        pointScatteringTarget.iX = 25;
        pointScatteringTarget.iY = 32;        
    }
    else if(iObjID == PME_OBJECT_GHOST_ORANGE)
    {
        iSpeed = PME_GHOST_START_SPEED;
        iGoingOutTrigger = 30;
        pointScatteringTarget.iX = 0;
        pointScatteringTarget.iY = 32;        
    }
    else
    {
        Main::Instance().ILogMgr().get()->msg(LML_CRITICAL, "     [Ghost] Ghost ID '%d' unknown, unexpected results!\n", iID);
        iGoingOutTrigger = -1;
    }         

    // Create our state objects
    pStateInit = new(std::nothrow) GhostStateInit("Init");
    pStateEvading = new(std::nothrow) GhostStateEvading("Evading");
    pStateChasing = new(std::nothrow) GhostStateChasing("Chasing");
    pStateScattering = new(std::nothrow) GhostStateScattering("Scattering");
    pStateDeath = new(std::nothrow) GhostStateDeath("Death");

    #ifdef DEBUG_INTERNAL
        Main::Instance().ILogMgr().get()->msg(LML_LOW, "     [Ghost] Ghost ID '%d' created.\n", iID);
    #endif
}

// Ghost destructor.
Ghost::~Ghost()
{
    delete pStateInit;
    pStateInit = nullptr;
    delete pStateEvading;
    pStateEvading = nullptr;
    delete pStateChasing;
    pStateChasing = nullptr;
    delete pStateScattering;
    pStateScattering = nullptr;
    delete pStateDeath;
    pStateDeath = nullptr;
    #ifdef DEBUG_INTERNAL
        Main::Instance().ILogMgr().get()->msg(LML_LOW, "     [Ghost] Ghost ID '%d' deleted.\n", iID);
    #endif
}

// Ghost logic execution.
// Return PME_LOOP
Sint32 Ghost::execute()
{   
    // Execute current state.
    // Note the states have full access to all our methods/attributes
    if(pCurrentState) pCurrentState->execute(this);
    
    // Call base method (set the object sprite position)
    Actor::execute();

    return PME_LOOP;
}

// Debug Ghost object.
#define PME_DEBUG_NUM_GHOST_ATTRIBUTES 6
Sint32 Ghost::debug(Sint32 iMode)
{
    Main &mC64 = Main::Instance();
    string sDebug, sTmp;
    Font* pFont = Main::Instance().IFontMgr().get(ResourceManager::Instance().get(RM_FONT_CONSOLE));
    
    // Call our base class (render target sprite)
    Actor::debug(iMode);

    // Start coordinates for each ghost
    Sint32 iDebugStartY = 0;
    switch(iID)
    {
    case PME_OBJECT_GHOST_RED:
        iDebugStartY = PME_DEBUG_PANEL_GHOST_Y + iMazeRenderingY;
        break;
    case PME_OBJECT_GHOST_BLUE:
        iDebugStartY = PME_DEBUG_PANEL_GHOST_Y + iMazeRenderingY + 1 * ((PME_DEBUG_NUM_GHOST_ATTRIBUTES * 16) + 16);
        break;
    case PME_OBJECT_GHOST_PINK:
        iDebugStartY = PME_DEBUG_PANEL_GHOST_Y + iMazeRenderingY + 2 * ((PME_DEBUG_NUM_GHOST_ATTRIBUTES * 16) + 16);
        break;
    case PME_OBJECT_GHOST_ORANGE:
        iDebugStartY = PME_DEBUG_PANEL_GHOST_Y + iMazeRenderingY + 3 * ((PME_DEBUG_NUM_GHOST_ATTRIBUTES * 16) + 16);
        break;
    }

    // Ghost information
    pFont->setPosition(PME_DEBUG_PANEL_GHOST_X, static_cast<float>(iDebugStartY));
    getName(sDebug);
    sDebug += " info";
    pFont->render(sDebug);

    // Maze and pixel position
    mC64.ITool().intToStrDec(iMazeX, sTmp);
    sDebug = "Position: [";
    sDebug += sTmp;
    sDebug += ",";
    mC64.ITool().intToStrDec(iMazeY, sTmp);
    sDebug += sTmp;
    sDebug += "] (";
    mC64.ITool().intToStrDec(iPositionX, sTmp);
    sDebug += sTmp;
    sDebug += ",";
    mC64.ITool().intToStrDec(iPositionY, sTmp);
    sDebug += sTmp;
    sDebug += ")";
    pFont->setPosition(PME_DEBUG_PANEL_GHOST_X, static_cast<float>(iDebugStartY + 16));
    pFont->render(sDebug);

    // Speed and GoingOutTrigger
    sDebug = "Speed: ";
    mC64.ITool().intToStrDec(iSpeed, sTmp);
    sDebug += sTmp; sDebug += " - GoingOut: ";
    mC64.ITool().intToStrDec(iGoingOutTrigger, sTmp);
    sDebug += sTmp;
    pFont->setPosition(PME_DEBUG_PANEL_GHOST_X, static_cast<float>(iDebugStartY + 32));
    pFont->render(sDebug);

    // Target        
    mC64.ITool().intToStrDec(pointTarget.iX, sTmp);
    sDebug = "Target: [";
    sDebug += sTmp;
    sDebug += ",";
    mC64.ITool().intToStrDec(pointTarget.iY, sTmp);
    sDebug += sTmp;
    sDebug += "]";
    pFont->setPosition(PME_DEBUG_PANEL_GHOST_X, static_cast<float>(iDebugStartY + 48));
    pFont->render(sDebug);    

    // State
    pCurrentState->getName(sDebug);
    sDebug = "State: " + sDebug;
    pFont->setPosition(PME_DEBUG_PANEL_GHOST_X, static_cast<float>(iDebugStartY + 64));
    pFont->render(sDebug);

    // Brain name
    if(BrainsFactory::Instance().getName(idBrain, sDebug) == -1) sDebug = "Empty";
    sDebug = "Brain: " + sDebug;
    pFont->setPosition(PME_DEBUG_PANEL_GHOST_X, static_cast<float>(iDebugStartY + 80));
    pFont->render(sDebug);

    return 0;
}

// Message for going to Init state.
Sint32 Ghost::msgGoInit()
{
    // Call base method
    Actor::msgGoInit();
    stateChange(pStateInit);
    return 0;
}

// Ghosts react changing to evading state.
Sint32 Ghost::msgPelletPowerEaten(Sint32 iX, Sint32 iY)
{    
    // Save temporal evasion target (current position of PacMan). 
    // It is transformed to a valid position on the evading state
    pointEvadingTarget.iX = iX;
    pointEvadingTarget.iY = iY;

    // Do not affect us if we are on Init or Death states
    if(!stateCurrentCheck(*pStateInit) && !stateCurrentCheck(*pStateDeath)) stateChange(pStateEvading);
    
    return 0;
}

// Ghost react changing to death state.
// return 0: eaten by PacMan. Going death state.
// return 1: we eat PacMan.
// return 2: already death, nothing to do but return home
Sint32 Ghost::msgGhostCollision()
{
    // If I am already death, do nothing
    if(stateCurrentCheck(*pStateDeath)) return 2;

    // If I am evading, I was eaten by PacMan
    if(stateCurrentCheck(*pStateEvading))
    {
        stateChange(pStateDeath);
        return 0;
    }

    // Otherwise, we eat PacMan!. We will receive a msgPause() for avoiding all movement till a msgGoInit() arrives.
    return 1;
}

// Get Scattering predefined target
Sint32 Ghost::getScatteringTarget(Sint32& iX, Sint32& iY)
{
    iX = pointScatteringTarget.iX;
    iY = pointScatteringTarget.iY;
    return 0;
}

// Given a desired target (could be valid or not), we try to reach it applying our rules:
// - try to reach target using euclidean distance
// - no go back (but when a state change it can happen)
// - keep moving
// - result is stored on vPath
Sint32 Ghost::applyMovementRules(Sint32 iTX, Sint32 iTY)
{
    Sint32 i, iSelected = 0;
    double dLowest = 1000.0f;
    vector<MazePoint> vMovementOptions;
  
    // 1.Get possible movement options. ToDO: tunnels?
    pGameField->getMovementOptions(iMazeX, iMazeY, vMovementOptions);

    // 2.Normal mode: can not go back
    for(i = 0; i < vMovementOptions.size();)
    {
        if((vMovementOptions[i].iX == iMazePrevX) && (vMovementOptions[i].iY == iMazePrevY))
        {            
            vMovementOptions.erase(vMovementOptions.begin() + i);
        }
        else ++i;
    }

    // 3.1.If there is only a possible way, avoid more calcs
    if(vMovementOptions.size() == 1) iSelected = 0;

    // 3.2.Assign distances to target and select the closest one.
    // In case of match, we select the first one: it is already prioritized in getMovementOptions()
    else
    {
        for(i = 0; i < vMovementOptions.size(); ++i) euclideanDistance(iTX, iTY, vMovementOptions[i]);
        for(i = 0; i < vMovementOptions.size(); ++i)
        {
            if(dLowest > vMovementOptions[i].dDistance)
            {
                dLowest = vMovementOptions[i].dDistance;
                iSelected = i;
            }
        }
    }

    // 4.Add the selected point to the path
    vPath.push_back(vMovementOptions[iSelected]);    
    return 0;
}

// Init State
// Wait for PME_GETREADY_TIME and then for our starting signal(depending on the number of eaten pellets)
GhostStateInit::GhostStateInit(const string& sN) : State(sN) { iTime = 0; }
void GhostStateInit::enter(Actor* pAct)
{
    Ghost* pGhost = reinterpret_cast<Ghost*>(pAct);

    // Call our base class method
    State::enter(pAct);

    // Reset our movemements vars    
    pGhost->eAM = Actor::eActorMoving::AM_NEXT_STEP;
    pGhost->vPath.clear();

    // Time for waiting in the ghost home but when we came from a death (eaten by PacMan), just wait 1/4. This way it is easier for PacMan    
    iTime = Main::Instance().ITimer().getTicksNow();
    if(pGhost->statePreviousCheck(*pGhost->pStateDeath)) iTime = iTime - static_cast<Uint64>(pGhost->pGameField->messageGetReady(-1) / 4);
}
void GhostStateInit::execute(Actor* pAct)
{  
    Ghost* pGhost = reinterpret_cast<Ghost*>(pAct);
    Sint32 iTmp;
    
    // 1.If we received a msgPause(), we avoid to execute anything else till the msgGoInit()
    if(pGhost->eAM == Actor::eActorMoving::AM_DISABLED) return;

    // 2.Before going out of home, wait for the getready time
    if(Main::Instance().ITimer().getTicksNow() < (iTime + pGhost->pGameField->messageGetReady(-1))) return;
    
    // 3.Are we waiting on the ghost home? Then move side to side
    if(pGhost->pGameField->getEatenPelletsPercent() < pGhost->iGoingOutTrigger)
    {
        // Have we reach our current target? Go to the other side (11,14) <-> (16,14)
        if(pGhost->vPath.size() == 0)
        {
            if(pGhost->iMazeX == 11) iTmp = 16;
            else iTmp = 11;
            pGhost->pGameField->mapSearch().findPath(pGhost->iMazeX, pGhost->iMazeY, iTmp, 14, pGhost->vPath, PME_STATE_WALKABLE_GHOST);
            pGhost->pointTarget.iX = iTmp;
            pGhost->pointTarget.iY = 14;
        }
        return;
    }

    // 4.Time to exit home. Wait for next wall hit
    if(pGhost->vPath.size() == 0)
    {
        pGhost->pGameField->mapSearch().findPath(pGhost->iMazeX, pGhost->iMazeY, 13, 11, pGhost->vPath, PME_STATE_WALKABLE_GHOST);
        pGhost->pointTarget.iX = 13;
        pGhost->pointTarget.iY = 11;
    }
    
    // 5.Have we reached exit target? then change to chasing
    if(pGhost->iMazeX == 13 && pGhost->iMazeY == 11)
    {
        pGhost->stateChange(pGhost->pStateChasing);
    }
}

// Evading State
GhostStateEvading::GhostStateEvading(const string& sN) : State(sN) { }
void GhostStateEvading::enter(Actor* pAct)
{
    Ghost* pGhost = reinterpret_cast<Ghost*>(pAct);
   
    // Call our base class method
    State::enter(pAct);
        
    // Slowdown the ghost
    pGhost->iSpeed -= PME_GHOST_SPEED_VARIATION;

    // Set first stage evading sprite
    pGhost->eSS = Actor::eSpriteState::SS_EVADE_FIRST_STAGE;
}
void GhostStateEvading::execute(Actor* pAct)
{
    Ghost* pGhost = reinterpret_cast<Ghost*>(pAct);
    Sint32 iX, iY, iWaveMode;

    // 1.When the empowered time expires...
    iWaveMode = pGhost->pGameField->getWaveMode();
    if((iWaveMode != PME_GLOBAL_WAVE_EVADING) && (iWaveMode != PME_GLOBAL_WAVE_EVADING_END))
    {
        // go back to previous state
        pGhost->statePrevious();
        return;
    }        

    // 2.When 1/4 of ticks left.. 
    if(iWaveMode == PME_GLOBAL_WAVE_EVADING_END)
    {
        // change to second stage evading sprite
        pGhost->eSS = Actor::eSpriteState::SS_EVADE_SECOND_STAGE;
    }

    // 3.Evasion!    
    if(pGhost->vPath.size() == 0)
    {
        // pointEvadingTarget has the coordinates were PacMan ate a power pellet so we try to target a distant point
        iX = Main::Instance().ITool().randWELL() % 6;
        iX = MAZE_WIDTH - pGhost->pointEvadingTarget.iX + iX - 3; // Randomize a bit the destination
        iY = Main::Instance().ITool().randWELL() % 6;
        iY = MAZE_HEIGHT - pGhost->pointEvadingTarget.iY + iY - 3; // Randomize a bit the destination
        pGhost->pGameField->validateMazePosition(iX, iY); // Validate the maze position, it will find the closest valid position
        pGhost->pGameField->mapSearch().findPath(pGhost->iMazeX, pGhost->iMazeY, iX, iY, pGhost->vPath, PME_STATE_WALKABLE);
        pGhost->pointTarget.iX = iX;
        pGhost->pointTarget.iY = iY;
    }
}
void GhostStateEvading::exit(Actor* pAct)
{    
    Ghost* pGhost = reinterpret_cast<Ghost*>(pAct);

    // Call our base class method
    State::exit(pAct);

    // Recover previous speed
    pGhost->iSpeed += PME_GHOST_SPEED_VARIATION;

    // Set default sprite state
    pGhost->eSS = Actor::eSpriteState::SS_DEFAULT;
    
    // Clear movement attributes
    pGhost->eAM = Actor::eActorMoving::AM_NEXT_STEP;
    pGhost->vPath.clear();
}

// Chasing State
GhostStateChasing::GhostStateChasing(const string& sN) : State(sN) { iTX = iTY = iITX = iITY = 0; bGetNewTarget = true; }
void GhostStateChasing::enter(Actor* pAct)
{
    // Call our base class method
    State::enter(pAct);
}
void GhostStateChasing::execute(Actor* pAct)
{
    vector<MazePoint> vIntersection;
    Ghost* pGhost = reinterpret_cast<Ghost*>(pAct);
    Sint32 iWaveMode;

    // 1.If we received a msgPause(), we avoid to execute anything else till the msgGoInit()
    if(pGhost->eAM == Actor::eActorMoving::AM_DISABLED) return;

    // 2.Check global wave mode
    iWaveMode = pGhost->pGameField->getWaveMode();
    if(iWaveMode == PME_GLOBAL_WAVE_SCATTERING)
    {
        pGhost->stateChange(pGhost->pStateScattering);
        return;
    }
    else if(iWaveMode == PME_GLOBAL_WAVE_EVADING)
    {
        pGhost->stateChange(pGhost->pStateEvading);
        return;
    }

    // 3.1.When we reach our target, request to get a new target
    if((pGhost->iMazeX == iTX) && (pGhost->iMazeY == iTY)) bGetNewTarget = true;

    // 3.2.At each intersection, request once to get a new target
    pGhost->pGameField->getMovementOptions(pGhost->iMazeX, pGhost->iMazeY, vIntersection);
    if((vIntersection.size() > 2) && ((pGhost->iMazeX != iITX) || (pGhost->iMazeY != iITY)))
    {
        iITX = pGhost->iMazeX;
        iITY = pGhost->iMazeY;
        bGetNewTarget = true;
    }

    // 3.3.Call to the ghost brain for getting a new "High-level target" in (iTX,iTY)
    if(bGetNewTarget)
    {
        pGhost->think(iTX, iTY);        
        pGhost->pointTarget.iX = iTX;
        pGhost->pointTarget.iY = iTY;
        bGetNewTarget = false;
    }

    // 4.Keep filling our path applying ghosts movement rules.
    if(pGhost->vPath.size() == 0)
    {           
        pGhost->applyMovementRules(iTX, iTY);
    }
}
void GhostStateChasing::exit(Actor* pAct)
{
    Ghost* pGhost = reinterpret_cast<Ghost*>(pAct);

    // Call our base class method
    State::exit(pAct);

    // Clear movement attributes
    pGhost->eAM = Actor::eActorMoving::AM_NEXT_STEP;
    pGhost->vPath.clear();
}

// Scattering State
GhostStateScattering::GhostStateScattering(const string& sN) : State(sN) {}
void GhostStateScattering::enter(Actor* pAct)
{
    // Call our base class method
    State::enter(pAct);
}
void GhostStateScattering::execute(Actor* pAct)
{    
    Sint32 iWaveMode;
    Ghost* pGhost = reinterpret_cast<Ghost*>(pAct);
    
    // 1.If we received a msgPause(), we avoid to execute anything else till the msgGoInit()
    if(pGhost->eAM == Actor::eActorMoving::AM_DISABLED) return;

    // 2.Check global wave mode
    iWaveMode = pGhost->pGameField->getWaveMode();
    if(iWaveMode == PME_GLOBAL_WAVE_CHASING)
    {
        pGhost->stateChange(pGhost->pStateChasing);
        return;
    }
    else if(iWaveMode == PME_GLOBAL_WAVE_EVADING)
    {
        pGhost->stateChange(pGhost->pStateEvading);
        return;
    }
    
    // 3.With no path, define a new target and try to reach it. Apply Ghost movement rules.
    if(pGhost->vPath.size() == 0)
    {
        pGhost->pointTarget = pGhost->pointScatteringTarget;
        pGhost->applyMovementRules(pGhost->pointScatteringTarget.iX, pGhost->pointScatteringTarget.iY);
    }
}
void GhostStateScattering::exit(Actor* pAct)
{
    Ghost* pGhost = reinterpret_cast<Ghost*>(pAct);

    // Call our base class method
    State::exit(pAct);

    // Clear movement attributes
    pGhost->eAM = Actor::eActorMoving::AM_NEXT_STEP;
    pGhost->vPath.clear();
}

// Death State
GhostStateDeath::GhostStateDeath(const string& sN) : State(sN) {}
void GhostStateDeath::enter(Actor* pAct)
{
    Ghost* pGhost = reinterpret_cast<Ghost*>(pAct);

    // Call our base class method
    State::enter(pAct);

    // Maximum ghost speed = PacMan speed
    pGhost->iSpeed = PME_PACMAN_START_SPEED;

    // Set death sprite
    pGhost->eSS = Actor::eSpriteState::SS_DEATH;
}
void GhostStateDeath::execute(Actor* pAct)
{
    Ghost* pGhost = reinterpret_cast<Ghost*>(pAct);

    // 1.If we received a msgPause(), we avoid to execute anything else till the msgGoInit()
    if(pGhost->eAM == Actor::eActorMoving::AM_DISABLED) return;

    // 2.Go home, starting position
    if(pGhost->vPath.size() == 0)
    {
        // 1.1.Have we reached our home?
        if(pGhost->iMazeX == (pGhost->iStartingPositionX / MAZE_TILE_SIZE) && pGhost->iMazeY == (pGhost->iStartingPositionY / MAZE_TILE_SIZE))
            pGhost->stateChange(pGhost->pStateInit);

        // 1.2.Get the path to our home
        else
        {
            pGhost->pointTarget.iX = pGhost->iStartingPositionX / MAZE_TILE_SIZE;
            pGhost->pointTarget.iY = pGhost->iStartingPositionY / MAZE_TILE_SIZE;
            pGhost->pGameField->mapSearch().findPath(pGhost->iMazeX, pGhost->iMazeY, pGhost->pointTarget.iX, pGhost->pointTarget.iY, pGhost->vPath, PME_STATE_WALKABLE_GHOST);
        }
    }
}
void GhostStateDeath::exit(Actor* pAct)
{
    Ghost* pGhost = reinterpret_cast<Ghost*>(pAct);

    // Call our base class method
    State::exit(pAct);

    // Recover our speed
    pGhost->iSpeed = PME_GHOST_START_SPEED;

    // Set default sprite state
    pGhost->eSS = Actor::eSpriteState::SS_DEFAULT;
}