/*----------------------------------------------------------------------
Pac-Man Evolution - Roberto Prieto
 Copyright (C) 2018-2025 MegaStorm Systems
contact@megastormsystems.com - http://www.megastormsystems.com

This software is provided 'as-is', without any express or implied
warranty.  In no event will the authors be held liable for any damages
arising from the use of this software.

Permission is granted to anyone to use this software for any purpose,
including commercial applications, and to alter it and redistribute it
freely, subject to the following restrictions:

1. The origin of this software must not be misrepresented; you must not
claim that you wrote the original software. If you use this software
in a product, an acknowledgment in the product documentation would be
appreciated but is not required.
2. Altered source versions must be plainly marked as such, and must not be
misrepresented as being the original software.
3. This notice may not be removed or altered from any source distribution.

------------------------------------------------------------------------

Objects, State and Actor classes

------------------------------------------------------------------------ */

#ifndef OBJECTSPME_H
#define OBJECTSPME_H

//#define DEBUG_FSM

#define PME_GETREADY_TIME          2000
#define PME_PACMAN_START_SPEED        4
#define PME_PACMAN_START_LIFES        3
#define PME_PACMAN_ENPOWERED_TIME  7000
#define PME_GHOST_START_SPEED         3
#define PME_GHOST_SPEED_VARIATION     1
#define PME_GHOST_CHASING_TIME    20000 //20000
#define PME_GHOST_SCATTERING_TIME  6000
#define PME_CRUISE_MODE               8  // After 8 changes, chasing is fixed

// Includes and forward definitions
#include "MemoryManager.h"
#include "Pac-Man_Evolution.h"
class GameField;
class Actor;

// ---------------- Object base class ----------------
class Object : public CMemPME
{
public:
    Object(Sint32, Sint32, Sint32, GameField*);
    virtual ~Object();	
    virtual Sint32 execute();
    Sint32 render(Sint32, Sint32);
    virtual Sint32 debug(Sint32);    

    // Messages that modified our state
    virtual Sint32 msgPause();
    virtual Sint32 msgGoInit(); 
    virtual Sint32 msgPelletPowerEaten(Sint32, Sint32);
    virtual Sint32 msgGhostCollision();
    
    // Generic methods
    Sint32 getID();
    void getName(string&);
    void getPositionMaze(Sint32&, Sint32&);
    void getDirection(Sint32&, Sint32&);

protected:	
    // Attributes    
    Sint32 iID;         // PME_ITEM_PELLET_POWER, PME_OBJECT_PACMAN, PME_OBJECT_GHOST_RED, etc
    Sint32 sprID;       // Object sprite ID    
    Sint32 iPositionX, iPositionY;  // Position on the maze in pixels
    Sint32 iMazeX, iMazeY;          // Position on the maze in tiles
    Sint32 iMazePrevX, iMazePrevY;  // Previous position on the maze in tiles
    Sint32 iStartingPositionX;      // Starting X position on the maze in pixels
    Sint32 iStartingPositionY;      // Starting Y position on the maze in pixels
    Sint32 iMazeRenderingX;         // X Position where the maze is rendered in pixels, calculated on Gamefield::render()
    Sint32 iMazeRenderingY;         // Y Position where the maze is rendered in pixels, calculated on Gamefield::render()
    bool bRenderExecuteSync;

    // Pointer to needed components
    GameField* pGameField;
};

// ---------------- State on the FSM ----------------
class State : public CMemPME
{
public:
    explicit State(const string&);
    virtual ~State();
    virtual void enter(Actor*);
    virtual void execute(Actor*) = 0;
    virtual void exit(Actor*);
    void getName(string &);
private:
    string sName;
};

// ---------------- Actor class ----------------
class Actor : public Object
{
public:
    Actor(Sint32 iObjID, Sint32 iMX, Sint32 iMY, GameField* GF);
    virtual ~Actor();
    virtual Sint32 execute();
    Sint32 debug(Sint32);
    
    enum eSpriteState
    {
        SS_DEFAULT = 0,
        SS_EVADE_FIRST_STAGE = 1,
        SS_EVADE_SECOND_STAGE = 2,
        SS_DEATH = 3
    };

    enum eActorMoving
    {
        AM_DISABLED = 0,
        AM_NEXT_STEP = 1,
        AM_WORKING = 2
    };

    Sint32 msgPause();
    Sint32 msgGoInit();
    
    // Auxiliary methods
    double euclideanDistance(Sint32, Sint32, MazePoint&);
    Sint32 getStateName(string&);

    // Brain management
    Sint32 getBrain();
    Sint32 setBrain(Sint32);
    Sint32 think(Sint32&, Sint32&);
    
protected:
    Sint32 moveX(Sint32);
    Sint32 moveY(Sint32);

    // Attributes          
    Sint32 iSpeed;              // Movement speed
    eSpriteState eSS;           // Sprite state modifier. 
    eActorMoving eAM;           // Moving status
    vector<MazePoint> vPath;    // Path that the actor will follow, first element is the target
    MazePoint pointNextStep;    // Next step coordinates 
    MazePoint pointTarget;      // Desired target
    Sint32 sprTargetID;         // Target sprite ID

    // Brain
    Sint32 idBrain;

    // State Machine methods and attributes for Pac-Man and Ghosts
    void stateChange(State*);
    void statePrevious();
    bool stateCurrentCheck(State&);
    bool statePreviousCheck(State&);
    State* pCurrentState;
    State* pPreviousState;       
};

#endif
