/*----------------------------------------------------------------------
Pac-Man Evolution - Roberto Prieto 
 Copyright (C) 2018-2025 MegaStorm Systems
contact@megastormsystems.com - http://www.megastormsystems.com

This software is provided 'as-is', without any express or implied
warranty.  In no event will the authors be held liable for any damages
arising from the use of this software.

Permission is granted to anyone to use this software for any purpose,
including commercial applications, and to alter it and redistribute it
freely, subject to the following restrictions:

1. The origin of this software must not be misrepresented; you must not
claim that you wrote the original software. If you use this software
in a product, an acknowledgment in the product documentation would be
appreciated but is not required.
2. Altered source versions must be plainly marked as such, and must not be
misrepresented as being the original software.
3. This notice may not be removed or altered from any source distribution.

------------------------------------------------------------------------

Object, State and Actor classes

------------------------------------------------------------------------ */

#include "Objects.h"
#include "GameField.h"
#include "ResourceManager.h"
#include "BrainsFactory.h"
#include <math.h>

// Object constructor.
Object::Object(Sint32 iObjID, Sint32 iMX, Sint32 iMY, GameField* GF)
{	    
    Sint32 j;
    float fSpeed;
    Main& mC64 = Main::Instance();
    Sprite* sprObj;
    
    // Inits vars
    iID = iObjID;
    iPositionX = iStartingPositionX = iMX * MAZE_TILE_SIZE;
    iPositionY = iStartingPositionY = iMY * MAZE_TILE_SIZE;
    iMazeX = iMazePrevX = iMX;
    iMazeY = iMazePrevY = iMY;
    iMazeRenderingX = iMazeRenderingY = 0;
    bRenderExecuteSync = false;

    // Pointer to needed components
    pGameField = GF;

    // Get a cloned sprite for this object. It should fit on the MAZE_TILE_SIZE x MAZE_TILE_SIZE. ToDO: autofit?
    switch(iObjID)
    {   
    case PME_OBJECT_PACMAN:
        sprID = ResourceManager::Instance().get(RM_SPR_PACMAN);
        break;
    case PME_OBJECT_GHOST_RED:
        sprID = ResourceManager::Instance().get(RM_SPR_GHOSTRED);
        break;
    case PME_OBJECT_GHOST_PINK:
        sprID = ResourceManager::Instance().get(RM_SPR_GHOSTPINK);
        break;
    case PME_OBJECT_GHOST_BLUE:
        sprID = ResourceManager::Instance().get(RM_SPR_GHOSTBLUE);
        break;   
    case PME_OBJECT_GHOST_ORANGE:
        sprID = ResourceManager::Instance().get(RM_SPR_GHOSTORANGE);
        break;
    case PME_OBJECT_PELLET_POWER1:
    case PME_OBJECT_PELLET_POWER2:
    case PME_OBJECT_PELLET_POWER3:
    case PME_OBJECT_PELLET_POWER4:
        sprID = ResourceManager::Instance().get(RM_SPR_PELLETPOWER);
        // Randomize the speed up to 10%
        sprObj = mC64.ISpriteMgr().get(sprID);
        j = mC64.ITool().randWELL() % 10;
        fSpeed = (float)j / 100.0f;
        fSpeed += 1.0f;
        sprObj->setSpeed(fSpeed);
        #ifdef DEBUG_INTERNAL
            mC64.ILogMgr().get()->msg(LML_NORMAL, "   [Object] Power Pellet speed set to '%.2f'.\n", fSpeed);
        #endif
        break;
    default:
        mC64.ILogMgr().get()->msg(LML_NORMAL, "   [Object] Unknown object id '%d'\n", iObjID);
    }
        
    #ifdef DEBUG_INTERNAL
        Main::Instance().ILogMgr().get()->msg(LML_LOW, "   [Object] Object ID '%d' created.\n", iID);
    #endif
}

// Object destructor.
Object::~Object()
{
    if(sprID != 0) Main::Instance().ISpriteMgr().close(sprID); // Remove our clon
    sprID = 0;   
    pGameField = nullptr;
    #ifdef DEBUG_INTERNAL
    Main::Instance().ILogMgr().get()->msg(LML_LOW, "   [Object] Object ID '%d' deleted.\n", iID);
    #endif
}

// Object logic execution. Interface method.
Sint32 Object::execute()
{
    if(bRenderExecuteSync) // Delay the execution till the first render() is performed
    {
        Main::Instance().ISpriteMgr().get(sprID)->setPosition(static_cast<float>(iMazeRenderingX + iPositionX), static_cast<float>(iMazeRenderingY + iPositionY), 1);
    }
    return 0;
}

// Object render. Interface method.
Sint32 Object::render(Sint32 iSX, Sint32 iSY)
{    
    iMazeRenderingX = iSX;
    iMazeRenderingY = iSY;
    if(!bRenderExecuteSync)
    {
        bRenderExecuteSync = true; // Synchronize render-execute
        execute();
    }
    if(sprID != 0) Main::Instance().ISpriteMgr().get(sprID)->render();
    return 0;
}

// Debug information. Not used in this object. Interface method.
Sint32 Object::debug(Sint32 iMode)
{
    return 0;
}

// Message sent when PacMan is death, pause all ghost movements. Not used in this object. Interface method.
Sint32 Object::msgPause()
{
    return 0;
}

// Message for going to Init state. Interface method.
Sint32 Object::msgGoInit()
{    
    iPositionX = iStartingPositionX;
    iPositionY = iStartingPositionY;
    iMazeX = iPositionX / MAZE_TILE_SIZE;
    iMazeY = iPositionY / MAZE_TILE_SIZE;
    return 0;
}

// Message sent when PacMan eats a power pellet. Not used in this object. Interface method.
Sint32 Object::msgPelletPowerEaten(Sint32 iX, Sint32 iY)
{
    return 0;
}

// Message sent when a PacMan/Ghost collision happens. Not used in this object. Interface method.
Sint32 Object::msgGhostCollision()
{
    return 0;
}

// Get ID.
Sint32 Object::getID()
{
    return iID;
}

// Get name
void Object::getName(string &sN)
{
    switch(iID)
    {
    case PME_OBJECT_PACMAN:
        sN = "PacMan";
        break;
    case PME_OBJECT_GHOST_RED:
        sN = "RedGhost";
        break;
    case PME_OBJECT_GHOST_PINK:
        sN = "PinkGhost";
        break;
    case PME_OBJECT_GHOST_BLUE:
        sN = "BlueGhost";
        break;
    case PME_OBJECT_GHOST_ORANGE:
        sN = "OrangeGhost";
        break;
    case PME_OBJECT_PELLET_POWER1:
        sN = "PowerPellet1";
        break;
    case PME_OBJECT_PELLET_POWER2:
        sN = "PowerPellet2";
        break;
    case PME_OBJECT_PELLET_POWER3:
        sN = "PowerPellet3";
        break;
    case PME_OBJECT_PELLET_POWER4:
        sN = "PowerPellet4";
        break;
    }
}

// Get maze position
void Object::getPositionMaze(Sint32 &iX, Sint32 &iY)
{
    iX = iMazeX;
    iY = iMazeY;
}

// Get direction. 
void Object::getDirection(Sint32 &iX, Sint32 &iY)
{
    Sint32 iState;
    
    // Default values
    iX = iY = 0;

    // Use current animation state of the sprite for querying the direction
    Main::Instance().ISpriteMgr().get(sprID)->getAnimState(-1, &iState);
    iState = SPR_GET_HIGHSTATE(iState);

    if(iState == SPR_STATE_UP) --iY;
    else if(iState == SPR_STATE_DOWN) ++iY;
    else if(iState == SPR_STATE_LEFT) --iX;
    else if(iState == SPR_STATE_RIGHT) ++iX;
}

// ---------------- State class ----------------
State::State(const string& sN)
{
    sName = sN;
}

State::~State() {}

void State::enter(Actor* pObj)
{
#ifdef DEBUG_FSM
    string sObjName;
    pObj->getName(sObjName);
    Main::Instance().ILogMgr().get()->msg(LML_INFO, "[%d] '%s' enters '%s' state\n", Main::Instance().ITimer().getTicksNow(), sObjName.c_str(), sName.c_str());
#endif
}

void State::exit(Actor* pObj)
{
#ifdef DEBUG_FSM
    string sObjName;
    pObj->getName(sObjName);
    Main::Instance().ILogMgr().get()->msg(LML_INFO, "[%d] '%s' exits '%s' state\n", Main::Instance().ITimer().getTicksNow(), sObjName.c_str(), sName.c_str());
#endif
}

void State::getName(string& sN)
{
    sN = sName;
}


// ---------------- Actor class ----------------
// Actor constructor.
Actor::Actor(Sint32 iObjID, Sint32 iMX, Sint32 iMY, GameField* GF) : Object(iObjID, iMX, iMY, GF)
{   
    Sprite* sprObj;

    // Init vars       
    iSpeed = 0;  
    eSS = SS_DEFAULT;
    eAM = AM_NEXT_STEP;
    vPath.clear();
    pointNextStep.iX = pointNextStep.iY = 0;
    pointTarget.iX = pointTarget.iY = 0;

    // Get a cloned target sprite for this object. It should fit on the MAZE_TILE_SIZE x MAZE_TILE_SIZE. ToDO: autofit?
    switch(iObjID)
    {
    case PME_OBJECT_PACMAN:
        sprTargetID = ResourceManager::Instance().get(RM_SPR_TARGETS);
        sprObj = Main::Instance().ISpriteMgr().get(sprTargetID);
        sprObj->selectAnim(4);
        break;
    case PME_OBJECT_GHOST_RED:
        sprTargetID = ResourceManager::Instance().get(RM_SPR_TARGETS);
        sprObj = Main::Instance().ISpriteMgr().get(sprTargetID);
        sprObj->selectAnim(0);
        break;
    case PME_OBJECT_GHOST_PINK:
        sprTargetID = ResourceManager::Instance().get(RM_SPR_TARGETS);
        sprObj = Main::Instance().ISpriteMgr().get(sprTargetID);
        sprObj->selectAnim(1);
        break;
    case PME_OBJECT_GHOST_BLUE:
        sprTargetID = ResourceManager::Instance().get(RM_SPR_TARGETS);
        sprObj = Main::Instance().ISpriteMgr().get(sprTargetID);
        sprObj->selectAnim(2);
        break;
    case PME_OBJECT_GHOST_ORANGE:
        sprTargetID = ResourceManager::Instance().get(RM_SPR_TARGETS);
        sprObj = Main::Instance().ISpriteMgr().get(sprTargetID);
        sprObj->selectAnim(3);
        break;
    default:
        sprTargetID = 0;
    }
    idBrain = -1;
    pCurrentState = nullptr;
    pPreviousState = nullptr;
    msgGoInit();

    #ifdef DEBUG_INTERNAL
        Main::Instance().ILogMgr().get()->msg(LML_LOW, "    [Actor] Actor ID '%d' created.\n", iID);
    #endif
}

// Actor destructor.
Actor::~Actor()
{    
    pCurrentState = nullptr;
    pPreviousState = nullptr;
    
    if(sprTargetID != 0) Main::Instance().ISpriteMgr().close(sprTargetID); // Remove our clon
    sprTargetID = 0;

    iSpeed = 0;
    eSS = SS_DEFAULT;
    
    #ifdef DEBUG_INTERNAL
        Main::Instance().ILogMgr().get()->msg(LML_LOW, "    [Actor] Actor ID '%d' deleted.\n", iID);
    #endif
}

// Execute the actor and perform all movement tasks
Sint32 Actor::execute()
{
    Sint32 iTmpX, iTmpY;

    // With no brain, exit
    if(idBrain == -1) return -1;

    // Call base method (set the object sprite position)
    Object::execute();

    // Follow the given path. The path must be a valid one.
    if(eAM != AM_DISABLED && vPath.size() > 0)
    {
        iTmpX = iMazeX;
        iTmpY = iMazeY;

        // Get next step
        if(eAM == AM_NEXT_STEP)
        {
            pointNextStep = vPath.back();
            eAM = AM_WORKING;
        }

        // Go to next step. Calculations is done in pixels
        if(eAM == AM_WORKING)
        {
            // Detect left tunnel
            if(iPositionX < (-(MAZE_TILE_SIZE / 2)))
            {
                pGameField->moveTo(iID, iMazeX, iMazeY, MAZE_WIDTH - 1, iMazeY);
                iMazeX = MAZE_WIDTH - 1;
                iPositionX = (iMazeX * MAZE_TILE_SIZE) + (MAZE_TILE_SIZE / 2);
                #ifdef DEBUG_INTERNAL
                Main::Instance().ILogMgr().get()->msg(LML_LOW, "tunel left\n");
                #endif
                pointNextStep.iX = iMazeX;
            }
            // Detect right tunnel
            else if(iPositionX >((MAZE_WIDTH - 1) * MAZE_TILE_SIZE) + (MAZE_TILE_SIZE / 2))
            {
                pGameField->moveTo(iID, iMazeX, iMazeY, 0, iMazeY);
                iMazeX = 0;
                iPositionX = -(MAZE_TILE_SIZE / 2);
                #ifdef DEBUG_INTERNAL
                Main::Instance().ILogMgr().get()->msg(LML_LOW, "tunel right\n");
                #endif
                pointNextStep.iX = iMazeX;
            }

            // Y axis
            if((pointNextStep.iY * MAZE_TILE_SIZE) > iPositionY)
            {
                moveY(1);
                // Control "overstep" as a state change could hit us in the middle of a tile
                if((pointNextStep.iY * MAZE_TILE_SIZE) < iPositionY)
                {
                    iPositionY = pointNextStep.iY * MAZE_TILE_SIZE;
                    #ifdef DEBUG_INTERNAL
                    Main::Instance().ILogMgr().get()->msg(LML_LOW, "Y+ overstep!\n");
                    #endif
                }
            }
            else if((pointNextStep.iY * MAZE_TILE_SIZE) < iPositionY)
            {
                moveY(-1);
                // Control "overstep" as a state change could hit us in the middle of a tile
                if((pointNextStep.iY * MAZE_TILE_SIZE) > iPositionY)
                {
                    iPositionY = pointNextStep.iY * MAZE_TILE_SIZE;
                    #ifdef DEBUG_INTERNAL
                    Main::Instance().ILogMgr().get()->msg(LML_LOW, "Y- overstep!\n");
                    #endif
                }
            }

            // X axis
            else if((pointNextStep.iX * MAZE_TILE_SIZE) > iPositionX)
            {
                moveX(1);
                // Control "overstep" as a state change could hit us in the middle of a tile
                if((pointNextStep.iX * MAZE_TILE_SIZE) < iPositionX)
                {
                    iPositionX = pointNextStep.iX * MAZE_TILE_SIZE;
                    #ifdef DEBUG_INTERNAL
                    Main::Instance().ILogMgr().get()->msg(LML_LOW, "X+ overstep!\n");
                    #endif
                }
            }
            else if((pointNextStep.iX * MAZE_TILE_SIZE) < iPositionX)
            {
                moveX(-1);
                // Control "overstep" as a state change could hit us in the middle of a tile
                if((pointNextStep.iX * MAZE_TILE_SIZE) > iPositionX)
                {
                    iPositionX = pointNextStep.iX * MAZE_TILE_SIZE;
                    #ifdef DEBUG_INTERNAL
                    Main::Instance().ILogMgr().get()->msg(LML_LOW, "X- overstep!\n");
                    #endif
                }
            }

            // We have reach our destination, go to next step
            else
            {
                eAM = AM_NEXT_STEP;
                vPath.pop_back();
            }
        }

        // Update previous maze position
        if(iTmpX != iMazeX || iTmpY != iMazeY)
        {
            iMazePrevX = iTmpX;
            iMazePrevY = iTmpY;
            iTmpX = iMazeX;
            iTmpY = iMazeY;
        }
    }

    return 0;
}

// Render the target sprite
Sint32 Actor::debug(Sint32 iMode)
{
    // Render target sprite
    if(iMode == 1)
    {
        Sprite* sprObj = Main::Instance().ISpriteMgr().get(sprTargetID);
        if(sprObj != nullptr)
        {
            Sint32 iX = pointTarget.iX;
            Sint32 iY = pointTarget.iY;

            // Perform a quick clipping as the target could be outside the screen and we want to render the target although not in the exact position
            if(iX < 0) iX = 0;
            else if(iX > MAZE_WIDTH) iX = MAZE_WIDTH - 1;
            if(iY < 0) iY = 0;
            else if(iY > MAZE_HEIGHT) iY = MAZE_HEIGHT - 1;

            // Set position and render it
            sprObj->setPosition(static_cast<float>(iMazeRenderingX + iX * MAZE_TILE_SIZE), static_cast<float>(iMazeRenderingY + iY * MAZE_TILE_SIZE), 0);
            sprObj->render();
        }
    }
    return 0;
}

// Stop actor movement.
Sint32 Actor::msgPause()
{
    eAM = AM_DISABLED;
    vPath.clear();
    return 0;
}

// Message for going Init state.
Sint32 Actor::msgGoInit()
{
    Object::msgGoInit(); // Call base method
    eSS = SS_DEFAULT;
    Main::Instance().ISpriteMgr().get(sprID)->selectAnim(SPR_STATE_LEFT);
    return 0;
}

// Return the Euclidean distance between given target coordinates and mazepoint(possible options)
double Actor::euclideanDistance(Sint32 iTX, Sint32 iTY, MazePoint& vMP)
{
    Sint32 iDX, iDY;
    iDX = abs(vMP.iX - iTX);
    iDY = abs(vMP.iY - iTY);
    vMP.dDistance = sqrt((iDX * iDX) + (iDY * iDY));
    return vMP.dDistance;
}

// Return current state name
Sint32 Actor::getStateName(string &sN)
{
    pCurrentState->getName(sN);
    return 0;
}

// Get brain id.
Sint32 Actor::getBrain()
{
    return idBrain;
}

// Set a new brain id
Sint32 Actor::setBrain(Sint32 idB)
{
    // Detect random brain and remove the object type
    if((idB & PME_BRAIN_TYPE_RANDOM) == PME_BRAIN_TYPE_RANDOM) idB = PME_BRAIN_TYPE_RANDOM;
    if(BrainsFactory::Instance().validate(idB) == true)
    {
        idBrain = idB;
        return 0;
    }
    return -1;
}

// Look for our brain and call it
Sint32 Actor::think(Sint32& iTX, Sint32& iTY)
{
    if(idBrain != -1)
    {
        return BrainsFactory::Instance().think(this, idBrain, iTX, iTY, pGameField);
    }
    return -1;
}

Sint32 Actor::moveX(Sint32 iDir)
{
    Sprite* sprObj = Main::Instance().ISpriteMgr().get(sprID);

    // Request RIGHT movement (pixel movement)
    if(iDir > 0)
    {
        // Pixel movement
        iPositionX += iSpeed;
        if(((iPositionX + (MAZE_TILE_SIZE / 2)) / MAZE_TILE_SIZE) != iMazeX)
        {
            // Changing to a new tile
            ++iMazeX;
            pGameField->moveTo(iID, iMazeX - 1, iMazeY, iMazeX, iMazeY);
        }

        // Set sprite animation
        sprObj->selectAnim(SPR_STATE_RIGHT + eSS);
    }

    // Request LEFT movement (pixel movement)
    else
    {
        // Pixel movement
        iPositionX -= iSpeed;
        if(((iPositionX + (MAZE_TILE_SIZE / 2)) / MAZE_TILE_SIZE) != iMazeX)
        {
            // Changing to a new tile
            --iMazeX;
            pGameField->moveTo(iID, iMazeX + 1, iMazeY, iMazeX, iMazeY);
        }

        // Set sprite animation
        sprObj->selectAnim(SPR_STATE_LEFT + eSS);
    }
    return 0;
}

Sint32 Actor::moveY(Sint32 iDir)
{
    Sprite* sprObj = Main::Instance().ISpriteMgr().get(sprID);

    // Request DOWN movement (pixel movement)
    if(iDir > 0)
    {
        // Pixel movement
        iPositionY += iSpeed;
        if(((iPositionY + (MAZE_TILE_SIZE / 2)) / MAZE_TILE_SIZE) != iMazeY)
        {
            // Changing to a new tile
            ++iMazeY;
            pGameField->moveTo(iID, iMazeX, iMazeY - 1, iMazeX, iMazeY);
        }

        // Set sprite animation
        sprObj->selectAnim(SPR_STATE_DOWN + eSS);
    }

    // Request UP movement (pixel movement)
    else
    {
        // Pixel movement
        iPositionY -= iSpeed;
        if(((iPositionY + (MAZE_TILE_SIZE / 2)) / MAZE_TILE_SIZE) != iMazeY)
        {
            // Changing to a new tile
            --iMazeY;
            pGameField->moveTo(iID, iMazeX, iMazeY + 1, iMazeX, iMazeY);
        }

        // Set sprite animation
        sprObj->selectAnim(SPR_STATE_UP + eSS);
    }
    return 0;
}

// Perform a state change
void Actor::stateChange(State* pNewState)
{
    // Is new state a valid one?
    if(!pNewState) return;

    // Keep previous state. Must be a different state.
    if(pNewState != pCurrentState) pPreviousState = pCurrentState;

    // Call the exit method of the existing state
    if(pCurrentState) pCurrentState->exit(this);

    // Change state to the new state
    pCurrentState = pNewState;

    // Call the entry method of the new state    
    pCurrentState->enter(this);
}

// Revert to previous state
void Actor::statePrevious()
{
    stateChange(pPreviousState);
}

// Check current state
bool Actor::stateCurrentCheck(State& pState)
{
    return typeid(*pCurrentState) == typeid(pState);
}

// Check previous state
bool Actor::statePreviousCheck(State& pState)
{
    if(pPreviousState == nullptr) return false;
    return typeid(*pPreviousState) == typeid(pState);
}
