/*----------------------------------------------------------------------
Pac-Man Evolution - Roberto Prieto
 Copyright (C) 2018-2025 MegaStorm Systems
contact@megastormsystems.com - http://www.megastormsystems.com

This software is provided 'as-is', without any express or implied
warranty.  In no event will the authors be held liable for any damages
arising from the use of this software.

Permission is granted to anyone to use this software for any purpose,
including commercial applications, and to alter it and redistribute it
freely, subject to the following restrictions:

1. The origin of this software must not be misrepresented; you must not
claim that you wrote the original software. If you use this software
in a product, an acknowledgment in the product documentation would be
appreciated but is not required.
2. Altered source versions must be plainly marked as such, and must not be
misrepresented as being the original software.
3. This notice may not be removed or altered from any source distribution.

------------------------------------------------------------------------

Hall of Fame class

------------------------------------------------------------------------ */

#include "HoF.h"
#include "Pac-Man_Evolution.h"
#include "ResourceManager.h"

HoF::HoF()
{	
    Sint32 i, j, idCDC;
    Main &mC64 = Main::Instance();
    Archive::Block myBlock;
    Archive *mCDC;

    // The scores are saved in an embedded file
    strcpy(myBlock.sParam1,"HoF.dat");
    strcpy(myBlock.sParam2, "C64_FILE");

    // First, try to load it
    memset(scores, 0, sizeof(scores));
    idCDC = mC64.IArchiveMgr().load(PME_CONFIG);
    mCDC = mC64.IArchiveMgr().get(idCDC);
    if(mCDC != nullptr)
    {
        if(mCDC->blockLoad(myBlock) == 0)
        {
            if(myBlock.iSize == sizeof(scores)) memcpy(scores, myBlock.pData, myBlock.iSize);
            mCDC->blockLoadReleaseData(myBlock);
        }
    }
    if(idCDC > 0) mC64.IArchiveMgr().close(idCDC);

    // If they are not present, we randomly generated them
    if(scores[0][0].iPoints == 0)
    {
        for(i = 0; i < 2; i++)
        {
            for(j = 0; j < PME_HOF_SCORES; j++)
            {
                switch(mC64.ITool().randMWC() % 8)
                {
                case 0:
                    strcpy(scores[i][j].sName, "Soraya");
                    break;
                case 1:
                    strcpy(scores[i][j].sName, "Lorena");
                    break;
                case 2:
                    strcpy(scores[i][j].sName, "Eduardo");
                    break;
                case 3:
                    strcpy(scores[i][j].sName, "Alvaro");
                    break;
                case 4:
                    strcpy(scores[i][j].sName, "Bob");
                    break;
                case 5:
                    strcpy(scores[i][j].sName, "Arturo");
                    break;
                case 6:
                    strcpy(scores[i][j].sName, "Diego");
                    break;
                case 7:
                    strcpy(scores[i][j].sName, "Shelly");
                    break;
                }
                scores[i][j].iPoints = (500 * PME_HOF_SCORES) - (j * 500);
            }
        }
    }
}

HoF::~HoF()
{
    Sint32 idCDC;
    Main &mC64 = Main::Instance();
    Archive::Block myBlock;
    Archive *mCDC;

    // The scores are saved in an embedded file
    strcpy(myBlock.sParam1, "HoF.dat");
    strcpy(myBlock.sParam2, "C64_FILE");

    // Save current scores
    idCDC = mC64.IArchiveMgr().load(PME_CONFIG);
    if(idCDC < 0) idCDC = mC64.IArchiveMgr().create(PME_CONFIG);
    mCDC = mC64.IArchiveMgr().get(idCDC);
    if(mCDC != NULL)
    {
        myBlock.iSize = sizeof(scores);
        myBlock.pData = scores;
        mCDC->blockAdd(myBlock);
    }
    if(idCDC > 0) mC64.IArchiveMgr().close(idCDC);
}

Sint32 HoF::store(Sint32 iGameType, Sint32 iPoints, char* szName)
{
    Sint32 i;

    // Security check of the gametype
    if(iGameType < PME_GAME_EVOLUTION) iGameType = 0;
    else iGameType = 1;

    for(i = 0; i < PME_HOF_SCORES; i++)
    {
        if(iPoints > scores[iGameType][i].iPoints)
        {
            insert(iGameType, i, iPoints, szName);
            break;
        }
    }       
    
    return 0;
}

// Get highest score of the given gametype
Sint32 HoF::getHighest(Sint32 iGameType)
{
    // Security check of the gametype
    if(iGameType < PME_GAME_EVOLUTION) iGameType = 0;
    else iGameType = 1;
    return scores[iGameType][0].iPoints;
}

// Get lowest score of the given gametype
Sint32 HoF::getLowest(Sint32 iGameType)
{
    // Security check of the gametype
    if(iGameType < PME_GAME_EVOLUTION) iGameType = 0;
    else iGameType = 1;
    return scores[iGameType][PME_HOF_SCORES - 1].iPoints;
}

// Render current HoF (position + name + elemental + points)
#define SCORES_X       100
#define SCORES_Y       280
#define SCORES_X_SPACE 440
#define SCORES_Y_SPACE  43
Sint32 HoF::render(Sint32 iMode)
{
    int iGT, i, iPos, iX, iLX, iY, iMaxPos, iMaxName, iMaxPoints;
    Main &mC64 = Main::Instance();
    Font* fontPlayer;
    Font* fontScore;
    string sTmp;
    SDL_FRect rDst;
    Sint32 iSW = 0, iSH = 0;
    Screen* pScreen = Main::Instance().IConfigMgr().get();
    if(pScreen != nullptr) pScreen->getSize(&iSW, &iSH);

    // Initial position
    iX = SCORES_X; iY = SCORES_Y;

    // Load score and air fonts
    fontScore = mC64.IFontMgr().get(ResourceManager::Instance().get(RM_FONT_SCORE));
    fontPlayer = mC64.IFontMgr().get(ResourceManager::Instance().get(RM_FONT_INFO));

    // Get the maximum "position text" size used for alignment purposes    
    iMaxPos = 0;
    for(iPos = 0; iPos < PME_HOF_SCORES; iPos++)
    {
        mC64.ITool().intToStrDec(iPos + 1, sTmp);
        i = fontScore->getWidth(sTmp);
        if(i > iMaxPos) iMaxPos = i;
    }

    // Render background centered
    Image* pImage = Main::Instance().IImageMgr().get(ResourceManager::Instance().get(RM_IMG_HOF));
    if(pImage != nullptr)
    {
        rDst.w = (float)pImage->getSurface()->w;
        rDst.h = (float)pImage->getSurface()->h;
        rDst.x = (float)(iSW - pImage->getSurface()->w) / 2.0f;
        rDst.y = (float)(iSH - pImage->getSurface()->h) / 2.0f;
        pImage->render(0, nullptr, &rDst);
    }
    
    // Render player names + points in the two different game types
    for(iGT = 0; iGT < 2; iGT++)
    {
        // Get the maximum "name text" and "points text" size used for aligment purposes
        // Assumes all custom fonts' players have the same size
        iMaxName = iMaxPoints = 0;
        for(iPos = 0; iPos < PME_HOF_SCORES; iPos++)
        {
            i = fontPlayer->getWidth(scores[iGT][iPos].sName);
            if(i > iMaxName) iMaxName = i;
            mC64.ITool().intToStrDec(scores[iGT][iPos].iPoints, sTmp);
            i = fontPlayer->getWidth(sTmp);
            if(i > iMaxPoints) iMaxPoints = i;
        }

        // Loop through all the scores rendering them aligned
        for(iPos = 0; iPos < PME_HOF_SCORES; iPos++)
        {
            iLX = iX;

            // Position (from 1 to HOF_SCORES)
            mC64.ITool().intToStrDec(iPos + 1, sTmp);
            fontPlayer->setPosition((float)(iLX - (fontScore->getWidth(sTmp) - iMaxPos)), (float)iY);
            fontPlayer->render(sTmp);
            iLX = iLX + iMaxPos + 10;

            // Player name
            fontPlayer->setPosition((float)(iLX - (fontPlayer->getWidth(scores[iGT][iPos].sName) - iMaxName)), (float)iY);
            fontPlayer->render(scores[iGT][iPos].sName);
            iLX = iLX + iMaxName + 15;

            // Player points
            mC64.ITool().intToStrDec(scores[iGT][iPos].iPoints, sTmp);
            fontScore->setPosition((float)(iLX - (fontScore->getWidth(sTmp) - iMaxPoints)), (float)iY);
            fontScore->render(sTmp);

            iY = iY + SCORES_Y_SPACE;
        }
        // Change the position for the next scores
        iX = iX + SCORES_X_SPACE;
        iY = SCORES_Y;
    }
    
    return PME_LOOP;
}

Sint32 HoF::insert(Sint32 iGameType, Sint32 iPos, Sint32 iPoints, char* szName)
{
    int i;

    // Before to insert the new player, we shift one position from the end to post
    for(i = PME_HOF_SCORES - 1; i > iPos; i--)
    {
        scores[iGameType][i].iPoints = scores[iGameType][i - 1].iPoints;
        strcpy(scores[iGameType][i].sName, scores[iGameType][i - 1].sName);
    }

    // Insert the new player
    scores[iGameType][iPos].iPoints = iPoints;
    strcpy(scores[iGameType][iPos].sName, szName);

    return 0;
}