/*----------------------------------------------------------------------
Pac-Man Evolution - Roberto Prieto
 Copyright (C) 2018-2025 MegaStorm Systems
contact@megastormsystems.com - http://www.megastormsystems.com

This software is provided 'as-is', without any express or implied
warranty.  In no event will the authors be held liable for any damages
arising from the use of this software.

Permission is granted to anyone to use this software for any purpose,
including commercial applications, and to alter it and redistribute it
freely, subject to the following restrictions:

1. The origin of this software must not be misrepresented; you must not
claim that you wrote the original software. If you use this software
in a product, an acknowledgment in the product documentation would be
appreciated but is not required.
2. Altered source versions must be plainly marked as such, and must not be
misrepresented as being the original software.
3. This notice may not be removed or altered from any source distribution.

------------------------------------------------------------------------

Genetic Algorithm class

Based on the amazing docs created by Mat Buckland (http://www.ai-junkie.com/ga/intro/gat1.html)

------------------------------------------------------------------------ */

#ifndef GENETICALGORITHM_H
#define GENETICALGORITHM_H

// Includes and forward definitions
#include "MemoryManager.h"

// Genome struct
struct Genome
{
    vector<double>	vWeights;
    float fFitness;

    Genome() { fFitness = 0; }
    Genome(vector<double> w, double f) { vWeights = w; fFitness = static_cast<float>(f); }

    // Overload '<' used for sorting
    friend bool operator<(const Genome& lhs, const Genome& rhs)
    {
        return(lhs.fFitness < rhs.fFitness);
    }
};

// Genetic algorithm class
class GeneticAlgorithm : public CMemPME
{
 public:
    // Constructor
    GeneticAlgorithm(Sint32, double, double, double, Sint32, Sint32, Sint32);

    // Initialize the algorithm
    void init(Sint32, vector<double>);

    // Evolve one generation
    vector<Genome> epoch(vector<Genome>&);

    // Access to attributes
    vector<Genome> getChromos()const;
    float fitnessAverage()const; 
    float fitnessBest()const;
    Sint32 getMutation()const;
    Sint32 getCrossOver()const;

private:
    // Hold all population's chromosomes (NN weights plus the fitness)
    vector<Genome> vPopulation;

    // Population and generation
    Sint32 iPopulation;
    Sint32 iGeneration;

    // Amount of weights per chromo
    Sint32 iChromoLength;

    // Fitness vars
    float fTotalFitness;
    float fFitnessBest;
    float fFitnessAverage;
    float fWorstFitness;

    // Keeps best genome
    Sint32	iFittestGenome;

    // Internal vars
    double dMutationRate;
    double dCrossoverRate;
    double dMaxPerturbation;
    Sint32 iNumElite, iNumCopiesElite;

    // Stats vars
    Sint32 iNumMutation;
    Sint32 iNumCrossOver;

    // Private methods doing the real job
    void crossover(const vector<double>&, const vector<double>&, vector<double>&, vector<double>&);
    void mutate(vector<double>&);
    Genome getChromoRoulette();
    void grabNBest(Sint32, const Sint32, vector<Genome>&);
    void calculateStats();
    void reset();
};

#endif

