/*----------------------------------------------------------------------
Pac-Man Evolution - Roberto Prieto
 Copyright (C) 2018-2025 MegaStorm Systems
contact@megastormsystems.com - http://www.megastormsystems.com

This software is provided 'as-is', without any express or implied
warranty.  In no event will the authors be held liable for any damages
arising from the use of this software.

Permission is granted to anyone to use this software for any purpose,
including commercial applications, and to alter it and redistribute it
freely, subject to the following restrictions:

1. The origin of this software must not be misrepresented; you must not
claim that you wrote the original software. If you use this software
in a product, an acknowledgment in the product documentation would be
appreciated but is not required.
2. Altered source versions must be plainly marked as such, and must not be
misrepresented as being the original software.
3. This notice may not be removed or altered from any source distribution.

------------------------------------------------------------------------

GameField class

------------------------------------------------------------------------ */

// Includes
#include "GameField.h"
#include "ResourceManager.h"
#include "MazeDynamic.h"
#include "MazeStatic.h"
#include "BrainsFactory.h"
#include "MapSearchAStar.h"
#include "Objects.h"
#include "ObjectsPacMan.h"
#include "ObjectsGhost.h"
#include <math.h>

// Constructor
GameField::GameField(GlobalStatus* GS)
{
    // Initialize variables
    sGlobalWave.reset();
    iMazeReady = iNumPellets = iNumEatenPellets = 0;
    iTimeStart = 0;
    iGetReadyTime = PME_GETREADY_TIME;
    iMazePixelX = iMazePixelY = 0;
    iFieldArray = nullptr;
    vObjects.clear();
    iRenderGraphicsStatus = -1;
    iSpecialTextCounter = -1;    
    sSpecialMessageCounter.clear();
    sSpecialMessage.clear();
    bDebug = bDebugDisableObjectRender = -1;
    bDebugShowTarget = 1;
    iDebugMode = 0;

    // Internal components
    pMapSearch = new(std::nothrow) MapSearchAStar(this);
    pMazeGen = nullptr;
    
    // Pointer to needed components    
    pGlobalStatus = GS;
    
    #ifdef DEBUG_INTERNAL
    Main::Instance().ILogMgr().get()->msg(LML_LOW, "  [Gamefield] Info: Object initialized.\n");
    #endif
}

// Destructor
GameField::~GameField()
{
    // Delete internal components
    if(pMapSearch != nullptr) delete pMapSearch;
    pMapSearch = nullptr;

    // Close
    close();

    #ifdef DEBUG_INTERNAL
    Main::Instance().ILogMgr().get()->msg(LML_LOW, "  [Gamefield] Info: Object closed.\n");
    #endif
}

// Load first maze of the given type and create the objects
Sint32 GameField::init()
{
    Actor* pActor = nullptr;
    Sint32 iPacManBrain, iGhostRedBrain, iGhostPinkBrain, iGhostBlueBrain, iGhostOrangeBrain;

    if(iMazeReady != 0) return PME_BREAK;
    
    // Workbench mode: brains assignment, disable: getReadyTime, scattering ghost mode and PacMan death animation
    if(pGlobalStatus->iGameType == PME_GAME_WORKBENCH)
    {
        iPacManBrain = pGlobalStatus->workBench.iPacManBrain;
        iGhostRedBrain = pGlobalStatus->workBench.iGhostRedBrain;
        iGhostPinkBrain = pGlobalStatus->workBench.iGhostPinkBrain;
        iGhostBlueBrain = pGlobalStatus->workBench.iGhostBlueBrain;
        iGhostOrangeBrain = pGlobalStatus->workBench.iGhostOrangeBrain;
        iGetReadyTime = 0;
        ResourceManager::Instance().setPacManDeathAnim(0);
    }
    // Standard and Evolution modes: brains assignment, enable: getReadyTime, scattering ghost mode and PacMan death animation
    else
    {
        // PacMan always with the human brain 
        iPacManBrain = PME_BRAIN_TYPE_HUMAN;
        // Standard game with fixed logic brains
        if(pGlobalStatus->iGameType == PME_GAME_STANDARD) iGhostRedBrain = iGhostPinkBrain = iGhostBlueBrain = iGhostOrangeBrain = PME_BRAIN_TYPE_FIXED;
        // Evolution game with EVN logic brains
        else iGhostRedBrain = iGhostPinkBrain = iGhostBlueBrain = iGhostOrangeBrain = PME_BRAIN_TYPE_EVOLVED;
        iGetReadyTime = PME_GETREADY_TIME;
        ResourceManager::Instance().setPacManDeathAnim(1);
    }

    // Create the field array
    if(iFieldArray == nullptr) iFieldArray = create2DArray<sField>(MAZE_HEIGHT, MAZE_WIDTH);

    // Create PacMan
    pActor = new(std::nothrow) PacMan(14, 23, this);
    pActor->setBrain(iPacManBrain | PME_OBJECT_PACMAN);    
    vObjects.push_back(pActor);

    // Create Red Ghost
    if(iGhostRedBrain != PME_OBJECT_NULL)
    {
        pActor = new(std::nothrow) Ghost(PME_OBJECT_GHOST_RED, 12, 14, this);
        pActor->setBrain(iGhostRedBrain | PME_OBJECT_GHOST_RED);        
        vObjects.push_back(pActor);
    }
    
    // Create Pink Ghost
    if(iGhostPinkBrain != PME_OBJECT_NULL)
    {
        pActor = new(std::nothrow) Ghost(PME_OBJECT_GHOST_PINK, 13, 14, this);
        pActor->setBrain(iGhostPinkBrain | PME_OBJECT_GHOST_PINK);
        vObjects.push_back(pActor);
    }
    
    // Create Blue Ghost
    if(iGhostBlueBrain != PME_OBJECT_NULL)
    {
        pActor = new(std::nothrow) Ghost(PME_OBJECT_GHOST_BLUE, 14, 14, this);
        pActor->setBrain(iGhostBlueBrain | PME_OBJECT_GHOST_BLUE);
        vObjects.push_back(pActor);
    }
    
    // Create Orange Ghost
    if(iGhostOrangeBrain != PME_OBJECT_NULL)
    {
        pActor = new(std::nothrow) Ghost(PME_OBJECT_GHOST_ORANGE, 15, 14, this);
        pActor->setBrain(iGhostOrangeBrain | PME_OBJECT_GHOST_ORANGE);
        vObjects.push_back(pActor);
    }

    #ifdef DEBUG_INTERNAL
        Main::Instance().ILogMgr().get()->msg(LML_NORMAL, "  [GameField] Info: Maze loaded and game initialized.\n");
    #endif
     
    // Initialize dynamic maze generator
    pMazeGen = new(std::nothrow) Maze(9, 5);

    // Load the next maze
    return nextMaze();
}

// Load next maze
Sint32 GameField::nextMaze()
{
    Sint32 x = 0, i, y = 0, iPowerPellet = PME_OBJECT_PELLET_POWER1;
    char* pMaze = nullptr;
    Object* pObj = nullptr;
    vector<Uint8> vTiles;

    if(iFieldArray == nullptr) return PME_BREAK;

    // Destroy power pellets objects (if applies)
    for(i = 0; i < vObjects.size();)
    {
        if(vObjects[i]->getID() >= PME_OBJECT_PELLET_POWER1)
        {
            delete vObjects[i];
            vObjects.erase(vObjects.begin() + i);
        }
        else ++i;
    }
    iNumPellets = iNumEatenPellets = 0;

    // Use a static maze
    if(pGlobalStatus->iGameType == PME_GAME_STANDARD)
    {
        if(iMazeStaticLength < (MAZE_WIDTH * MAZE_HEIGHT)) return PME_BREAK;
        i = iMazeReady % iMazeStaticNumber;
        pMaze = (char*)mazeStatic[i];
        iMazeDynamicGenerationAttemps = 1;
        iMazeDynamicGenerationTime = 0;
    }
    // Load dynamic and deterministic maze        
    else
    {
        i = static_cast<Sint32>(Main::Instance().ITimer().getTicksNow()); 
        iMazeDynamicGenerationAttemps = pMazeGen->createMaze(vTiles);        
        iMazeDynamicGenerationTime = static_cast<Sint32>(Main::Instance().ITimer().getTicksNow());
        iMazeDynamicGenerationTime -= i;
        pMaze = (char*)&vTiles[0];  
    }

    // Parse the maze
    for(y = 0; y < MAZE_HEIGHT; y++)
        for(x = 0; x < MAZE_WIDTH; x++)
        {
            // Default values
            iFieldArray[y][x].iState = PME_STATE_NULL;
            iFieldArray[y][x].iItem = PME_ITEM_NULL;
            iFieldArray[y][x].iObject = PME_OBJECT_NULL;

            // Parse current maze tile
            switch(*pMaze)
            {
                // External maze tile
            case '_':
                // Same as default values
                break;

                // Wall tile
            case '|':
                iFieldArray[y][x].iState = PME_STATE_WALL;
                break;

                // Walkable and empty tile
            case ' ':
                iFieldArray[y][x].iState = PME_STATE_WALKABLE;
                break;

                // Walkable and a pellet tile
            case '.':
                iFieldArray[y][x].iState = PME_STATE_WALKABLE;
                iFieldArray[y][x].iItem = PME_ITEM_PELLET;
                ++iNumPellets;
                break;

                // Walkable and a power pellet tile
            case 'o':
                iFieldArray[y][x].iState = PME_STATE_WALKABLE;
                if(pGlobalStatus->workBench.iTraining != 1)
                {                    
                    pObj = new(std::nothrow) Object(iPowerPellet, x, y, this);
                    vObjects.push_back(pObj);
                    iFieldArray[y][x].iObject = iPowerPellet;
                    iPowerPellet *= 2;
                    ++iNumPellets;
                }
                break;
            }            
            ++pMaze; // Next maze tile
        }

    // Ghost home's door and home initialization
    iFieldArray[12][13].iState = PME_STATE_WALKABLE_GHOST;
    iFieldArray[12][14].iState = PME_STATE_NULL;
    for(y = 13; y < 16; y++)
        for(x = 11; x < 17; x++)
        {
            // Default values
            iFieldArray[y][x].iState = PME_STATE_WALKABLE_GHOST;
        }

    // PacMan and ghosts initial position    
    initObjects();    

    // Music fade out and screen to black, then start playing new game music
    Main::Instance().IAudioTrackMgr().fadeOutTag(ATT_MUSIC, 500);
    Main::Instance().IConfigMgr().get()->fadeToColor(0, 0, 0, 500);
    Main::Instance().IAudioTrackMgr().get(ResourceManager::Instance().get(RM_MUS_GAME))->play(-1);
    pGlobalStatus->iRenderScreen = PME_SCREEN_GAME;

    ++iMazeReady;
    return PME_LOOP;
}

// Close current maze removing all resources and objects
Sint32 GameField::close()
{    
    Sint32 i;
        
    // Remove all objects
    for(i = 0; i < vObjects.size(); ++i) delete vObjects[i];
    
    // Reset vars
    iMazeReady = iNumPellets = iNumEatenPellets = 0;
    iTimeStart = 0;
    iMazePixelX = iMazePixelY = 0;
    if(iFieldArray) delete2DArray(iFieldArray);
    iFieldArray = nullptr;
    vObjects.clear();
    iRenderGraphicsStatus = -1;
    iSpecialTextCounter = -1;
    sSpecialMessageCounter.clear();
    sSpecialMessage.clear();

    // Delete our maze generator
    if(pMazeGen != nullptr) delete pMazeGen;
    pMazeGen = nullptr;

    // Music fade out and screen to black
    Main::Instance().IAudioTrackMgr().fadeOutTag(ATT_MUSIC, 500);
    Main::Instance().IConfigMgr().get()->fadeToColor(0, 0, 0, 500);

    #ifdef DEBUG_INTERNAL
        Main::Instance().ILogMgr().get()->msg(LML_NORMAL, "  [GameField] Info: Maze closed.\n\n");
    #endif  

    return 0;
}

// Main execution of the maze
// Return PME_MAZE_END when current maze is over and going to next one or PME_BREAK when game is aborted/finished (running out of lifes)
Sint32 GameField::execute()
{
    Sint32 iDone = PME_LOOP, i;
    SDL_Event eEvent;
    Main& mC64 = Main::Instance();

    // Is the maze ready?
    if(iMazeReady == 0) return PME_BREAK;
    
    // Display starting message
    if(pGlobalStatus->iGameType != PME_GAME_WORKBENCH) messageStart();
    
    // Main maze loop
    // mC64.ITimer().init(TS_RESET); // ToDO: reset stats but keep LFR, RFR but does not work
    iTimeStart = mC64.ITimer().getTicksNow();
    iRenderGraphicsStatus = RENDERGRAPHICS_GAME;    
    while(iDone == PME_LOOP)
    {
        // Update GlobalWave
        sGlobalWave.update();

        // Execution of our objects
        for(i = 0; (i < vObjects.size()) && (iDone == PME_LOOP) && (sSpecialMessage != "PAUSED"); ++i)
        {
            // Put more attention when PacMan is executed:
            // 1.Running out of lifes and the game is over
            // 2.Being on chasing state and the wave ticks is not modified
            if(vObjects[i]->getID() == PME_OBJECT_PACMAN)
            {
                // PacMan can return PME_ACTOR_ALIVE, PME_ACTOR_SPECIAL or PME_GAME_OVER
                iDone = vObjects[i]->execute();
                // Reduce a wave tick while PacMan is not in special mode
                if(iDone != PME_ACTOR_SPECIAL) --sGlobalWave.iTicks;
                else iDone = PME_LOOP; // Keep the main loop running
            }
            
            // Rest of objects execution. Always return PME_LOOP so dont care about it
            else vObjects[i]->execute();
        }        
       
        // Detect end of maze when no pellets left
        if(iNumPellets == iNumEatenPellets)
        {
            iDone = PME_MAZE_END;
        }

        // Detect end of assigned time
        if(pGlobalStatus->iGameType == PME_GAME_WORKBENCH)
        {
            if(pGlobalStatus->workBench.iTime != 0)
            {
                if((mC64.ITimer().getTicksNow() - iTimeStart) > (pGlobalStatus->workBench.iTime * 1000))
                {
                    iDone = PME_BREAK;
                }
            }
        }
               
        // Game event loop
        while(mC64.update(&eEvent))
        {
            switch(eEvent.type)
            {
            case SDL_EVENT_KEY_UP:
                switch(eEvent.key.key)
                {
                    // Finish the game
                case SDLK_ESCAPE:
                    iDone = PME_BREAK;
                    break;

                    // Pause the game
                case SDLK_SPACE: 
                    if(sSpecialMessage == "PAUSED") sSpecialMessage.clear();
                    else if(sSpecialMessage.size() == 0) sSpecialMessage = "PAUSED";
                    break;

                    // Enter debug mode
                case SDLK_F1: 
                    bDebug = bDebug * (-1);
                    break;
                case SDLK_F2:
                    if(bDebug == 1)
                    {
                        if(iDebugMode > 0) --iDebugMode;
                    }
                    break;
                case SDLK_F3:
                    if(bDebug == 1)
                    {
                        if(iDebugMode < 4) ++iDebugMode;
                    }
                    break;
                case SDLK_F4:
                    // Enable or disable objects rendering
                    if(bDebug == 1)
                    {
                        bDebugDisableObjectRender = bDebugDisableObjectRender * (-1);
                    }
                    break;                
                case SDLK_F5:
                    // Take a snapshot
                    if(bDebug == 1)
                    {
                        mC64.IConfigMgr().get()->getSnapshot("screenshot.png");
                    }
                    break;
                case SDLK_F6:
                    // Force to finish current maze and advance to the next one
                    if(bDebug == 1)
                    {
                        iDone = PME_MAZE_END;
                    }
                    break;
                case SDLK_F7:
                    // Enable or disable showing target sprite
                    if(bDebug == 1)
                    {
                        bDebugShowTarget = bDebugShowTarget * (-1);
                    }
                    break;
                case SDLK_F9:
                    vObjects[0]->msgGhostCollision(); // kill pacman TODO REMOVE
                    for(i = 0; i < vObjects.size(); ++i) vObjects[i]->msgPause();
                    break;
                }
                break;
            }
        }
    }
    
    // Detected game end
    if(iDone != PME_MAZE_END)
    {
        if(pGlobalStatus->iGameType != PME_GAME_WORKBENCH) messageEnd(iDone);
    }

    // Print out game info
    if(pGlobalStatus->workBench.iTraining != 1)
    {
        mC64.ILogMgr().get()->msg(LML_NORMAL, "  [GameField] Info: Maze '%d' (%s %d %dms)- %2.2f FPS during %2.2f seconds - Points %d \n",
            iMazeReady, (pGlobalStatus->iGameType == PME_GAME_STANDARD) ? "static" : "dynamic", iMazeDynamicGenerationAttemps, iMazeDynamicGenerationTime,
            mC64.ITimer().getAverageRFR(), (float)(mC64.ITimer().getTicksNow() - iTimeStart) / 1000.0f,
            pGlobalStatus->iPoints);
    }

    return iDone;
}

// Render graphics callback with different states
Sint32 GameField::render(Sint32 iMode)
{
    Sint32 iScreenW, iScreenH;
    Sint32 x, y, i, iLifes;
    SDL_FRect rDst;
    string sText, sTmp;
    Main& mC64 = Main::Instance();
    Font* pFont;
    Screen* pScreen;    

    // Is the maze ready?
    if(iMazeReady == 0) return PME_BREAK;

    // Get screen size and auto-adapt to screen changes. Get the starting coordinates for the maze rendering used by all objects.
    pScreen = mC64.IConfigMgr().get();
    pScreen->getSize(&iScreenW, &iScreenH);    
    iMazePixelX = (iScreenW - (MAZE_TILE_SIZE * MAZE_WIDTH)) / 2;
    iMazePixelY = MAZE_TILE_SIZE;
    
    // Sprite size equal to tile size
    rDst.w = rDst.h = (float)MAZE_TILE_SIZE;

    // Render black background 
    pScreen->clear();

    // Render game background    
    // ToDO: a starfield?
    
    // Render the "static" parts of the maze: walls and pellets
    for(y = 0; y < MAZE_HEIGHT; y++)
    {
        for(x = 0; x < MAZE_WIDTH; x++)
        {
            if(iFieldArray[y][x].iState == PME_STATE_WALL)
            {
                rDst.x = (x * rDst.w) + (float)iMazePixelX;
                rDst.y = (y * rDst.h) + (float)iMazePixelY;
                mC64.IGFX().rectFilled(rDst.x, rDst.y, rDst.x + rDst.w, rDst.y + rDst.h, 0x0000AAFF);
            }
            if(iFieldArray[y][x].iItem == PME_ITEM_PELLET)
            {
                rDst.x = (x * rDst.w) + (float)iMazePixelX;
                rDst.y = (y * rDst.h) + (float)iMazePixelY;
                i = ResourceManager::Instance().get(RM_SPR_PELLET);
                mC64.ISpriteMgr().get(i)->setPosition(rDst.x, rDst.y);
                mC64.ISpriteMgr().get(i)->render();
            }
        }
    }

    // Render objects: power pellets, ghosts and PacMan. Get lifes of PacMan
    for(i = (Sint32)vObjects.size() - 1; vObjects.size() > i; --i) // Reverse mode for rendering PacMan sprite over the rest of objects
    {
        if(vObjects[i]->getID() == PME_OBJECT_PACMAN)
        {
            iLifes = reinterpret_cast<PacMan*>(vObjects[i])->getLifes();
        }
        if(bDebugDisableObjectRender == -1) vObjects[i]->render(iMazePixelX, iMazePixelY);
    }
    
    // Render the score, highest score and pacman lifes
    pFont = mC64.IFontMgr().get(ResourceManager::Instance().get(RM_FONT_SCORE));
    pFont->setPosition((float)iMazePixelX, -8.0f);
    mC64.ITool().intToStrDec(pGlobalStatus->iPoints, sTmp);
    sText = "Score "; sText += sTmp;
    pFont->render(sText);
    rDst.x = (float)((iScreenW / 2) - MAZE_TILE_SIZE - (MAZE_TILE_SIZE / 2));
    rDst.y = 0.0f; rDst.w = (float)MAZE_TILE_SIZE; rDst.h = (float)MAZE_TILE_SIZE;
    while(iLifes > 0)
    {
        mC64.IImageMgr().get(ResourceManager::Instance().get(RM_IMG_ICON))->render(0, nullptr, &rDst);
        --iLifes;
        rDst.x += (float)MAZE_TILE_SIZE;
    }
    mC64.ITool().intToStrDec(pGlobalStatus->iHighestScore, sTmp);
    sText = "Highest "; sText += sTmp;
    pFont->setPosition((float)(iMazePixelX + (MAZE_WIDTH * MAZE_TILE_SIZE) - 250), -8.0f); 

    pFont->render(sText);

    // While we are in RENDERGRAPHICS_START status (at the starting of the maze)
    if(iRenderGraphicsStatus == RENDERGRAPHICS_START)
    {
        pScreen = mC64.IConfigMgr().get();
        pScreen->getSize(&i, nullptr);
        pFont = mC64.IFontMgr().get(ResourceManager::Instance().get(RM_FONT_INFO));

        specialText(pFont, (float)((i - pFont->getWidth(sSpecialMessage)) / 2), (float)((iScreenH / 2) - 150.0f), sSpecialMessage, iSpecialTextCounter);
        pFont->setPosition((float)((i - pFont->getWidth(sSpecialMessageCounter)) / 2), (float)((iScreenH / 2) - 80));
        pFont->render(sSpecialMessageCounter);
    }
    // Normal rendering game state
    else
    {
        // Render special message if present: get ready!
        if(sSpecialMessage.size() > 0)
        {
            pFont->setPosition((float)((iScreenW - pFont->getWidth(sSpecialMessage)) / 2), (float)((iScreenH / 2) + 20));
            pFont->render(sSpecialMessage);
        }

        // Debug information 
        if(bDebug == 1)
        {
            for(i = 0; i < vObjects.size(); ++i) vObjects[i]->debug(bDebugShowTarget);
            debug();
        }
    }
   
    return 0;
}

// Return a reference to the A* component
MapSearchAStar& GameField::mapSearch()
{
   return *pMapSearch;
}

// Get the state from the maze given x,y
Sint32 GameField::getState(Sint32 iMX, Sint32 iMY)
{
    // Check boundaries
    if(iMX < 0) return PME_STATE_NULL;
    else if(iMX > MAZE_WIDTH - 1) return PME_STATE_NULL;
    if(iMY < 0) return PME_STATE_NULL;
    else if(iMY > MAZE_HEIGHT - 1) return PME_STATE_NULL;

    // Get status
    return iFieldArray[iMY][iMX].iState;
}

// Get the item from the maze given x,y
Sint32 GameField::getItem(Sint32 iMX, Sint32 iMY)
{
    // Check boundaries
    if(iMX < 0) return PME_STATE_NULL;
    else if(iMX > MAZE_WIDTH - 1) return PME_STATE_NULL;
    if(iMY < 0) return PME_STATE_NULL;
    else if(iMY > MAZE_HEIGHT - 1) return PME_STATE_NULL;

    // Get status
    return iFieldArray[iMY][iMX].iItem;
}

// Get the percent of eaten pellets on this maze. Used for the ghosts going out of their home.
Sint32 GameField::getEatenPelletsPercent()
{
    return (iNumEatenPellets * 100) / iNumPellets;
}

// Get an object maze position.
Sint32 GameField::getObjectPosition(Sint32 iID, Sint32 &iX, Sint32 &iY)
{
    Sint32 i = getObjectIndex(iID);
    if(i != -1)
    {
        vObjects[i]->getPositionMaze(iX, iY);
        i = 0;
    }
    return i;
}

// Get an object direction
Sint32 GameField::getObjectDirection(Sint32 iID, Sint32 &iX, Sint32 &iY)
{
    Sint32 i = getObjectIndex(iID);
    if(i != -1)
    {
        vObjects[i]->getDirection(iX, iY);
        i = 0;
    }
    return i;
}

// Get an object state name
Sint32 GameField::getObjectStateName(Sint32 iID, string &sN)
{
    Sint32 i = getObjectIndex(iID);
    if(i != -1)
    {
        reinterpret_cast<Actor*>(vObjects[i])->getStateName(sN);
        i = 0;
    }
    return i;
}

// Return the closest pellet to the given object
Sint32 GameField::getClosestPellet(Sint32 iID, Sint32 &iX, Sint32 &iY)
{
    Sint32 iRadius = 1, iOX, iOY;
    Sint32 i = getObjectIndex(iID);
    if(i != -1)
    {
        vObjects[i]->getPositionMaze(iOX, iOY);
        
        // Check with iRadius[1,32]
        while(iRadius < 32)
        {
            for(int y = -iRadius; y <= iRadius; y++)
            {
                for(int x = -iRadius; x <= iRadius; x++)
                {
                    if(x*x + y * y <= iRadius * iRadius)
                    {
                        iX = iOX + x;
                        iY = iOY + y;
                        // ToDO: add power pellet too
                        if(getItem(iX, iY) == PME_ITEM_PELLET)
                        {
                            //printf("Radius %d (%d,%d)\n", iRadius, iX, iY);
                            return 0;
                        }
                    }
                }
            }
            ++iRadius;
        }

    }
    return i;
}

// Move an object to a new position. The position must be valid.
// Here we check for pellets, power pellets and ghost collisions
Sint32 GameField::moveTo(Sint32 iID, Sint32 iCurMX, Sint32 iCurMY, Sint32 iNewMX, Sint32 iNewMY)
{
    Sint32 i, iObjs, iTmp;
    Sint32 iX = 0, iY = 0;

    // Remove from old position
    iFieldArray[iCurMY][iCurMX].iObject -= iID;

    // Set to new position
    iFieldArray[iNewMY][iNewMX].iObject += iID;

    // Get objects on this new position
    iObjs = iFieldArray[iNewMY][iNewMX].iObject;

    // Only with PacMan, we check:
    if(iID == PME_OBJECT_PACMAN)
    {
        // Pellet item
        if(iFieldArray[iNewMY][iNewMX].iItem == PME_ITEM_PELLET)
        {
            iFieldArray[iNewMY][iNewMX].iItem = PME_ITEM_NULL;
            Main::Instance().IAudioTrackMgr().get(ResourceManager::Instance().get(RM_SND_GAMEEATPELLET))->play();
            addPoints(PME_POINTS_EAT_PELLET);
            ++iNumEatenPellets;
        }

        // Power Pellets        
        if(iObjs >= PME_OBJECT_PELLET_POWER1)
        {        
            iTmp = PME_GET_PELLET(iObjs); // Get only the power pellet id
            iFieldArray[iNewMY][iNewMX].iObject -= iTmp;
            Main::Instance().IAudioTrackMgr().get(ResourceManager::Instance().get(RM_SND_GAMEEATPELLETPOWER))->play();
            addPoints(PME_POINTS_EAT_PELLET_POWER);
            ++iNumEatenPellets;
            
            // Get coordinates for informing ghosts and remove power pellet object
            i = getObjectIndex(iTmp);
            if(i != -1)
            {
                vObjects[i]->getPositionMaze(iX, iY);
                delete vObjects[i];
                vObjects.erase(vObjects.begin() + i);
            }

            // PacMan and Ghosts react to a power pellet
            for(i = 0; i < vObjects.size(); ++i) vObjects[i]->msgPelletPowerEaten(iX, iY);
        }        
    }

    // Check for PacMan/Ghosts collisions
    if(iObjs & PME_OBJECT_PACMAN) // PacMan is on this position
    {
        iTmp = 2; // Assume no ghost collision

        // Check for each ghost: send the message only to the first one found (not in death state)
        if(iObjs & PME_OBJECT_GHOST_RED) // Red Ghost
        {   
            i = getObjectIndex(PME_OBJECT_GHOST_RED);
            if(i != -1) iTmp = vObjects[i]->msgGhostCollision();
        }
        else if((iObjs & PME_OBJECT_GHOST_BLUE) && (iTmp == 2)) // Blue Ghost
        {
            i = getObjectIndex(PME_OBJECT_GHOST_BLUE);
            if(i != -1) iTmp = vObjects[i]->msgGhostCollision();
        }
        else if((iObjs & PME_OBJECT_GHOST_PINK) && (iTmp == 2)) // Pink Ghost
        {
            i = getObjectIndex(PME_OBJECT_GHOST_PINK);
            if(i != -1) iTmp = vObjects[i]->msgGhostCollision();
        }
        else if((iObjs & PME_OBJECT_GHOST_ORANGE) && (iTmp == 2)) // Orange Ghost
        {
            i = getObjectIndex(PME_OBJECT_GHOST_ORANGE);
            if(i != -1) iTmp = vObjects[i]->msgGhostCollision();
        }       

        // With iTmp = 0, PacMan has eaten a ghost
        if(iTmp == 0)
        {
            i = getObjectIndex(PME_OBJECT_PACMAN);
            if(i != -1)
            {
                Main::Instance().IAudioTrackMgr().get(ResourceManager::Instance().get(RM_SND_GAMEEATGHOST))->play();
                addPoints(vObjects[i]->msgGhostCollision());
            }
        }
        // With iTmp = 1, PacMan was eaten by a ghost. Sent the pause message to all objects (ghosts)
        else if(iTmp == 1)
        {
            i = getObjectIndex(PME_OBJECT_PACMAN);
            if(i != -1)
            {
                if(pGlobalStatus->iGameType != PME_GAME_WORKBENCH) Main::Instance().IAudioTrackMgr().get(ResourceManager::Instance().get(RM_SND_GAMEPLAYERDEATH))->play();
                vObjects[i]->msgGhostCollision();
                for(i = 0; i < vObjects.size(); ++i) vObjects[i]->msgPause();               
            }
        }
        // With iTmp = 2 nothing happens (collision between PacMan and a death Ghost)
        
    }
    return 0;
}

// Show or hide the get ready message.
// Controlled by the PacMan while in Init state.
Sint32 GameField::messageGetReady(Sint32 iShow)
{
    if(iShow == 0) sSpecialMessage.clear();
    else if(iShow == 1) sSpecialMessage = "Get Ready!";
    return iGetReadyTime;
}

// Initialize the objects (PacMan and Ghosts) setting the start position and init state
Sint32 GameField::initObjects()
{
    Sint32 i, x, y;

    // Clean previous status and take care of power pellet objects! if a ghost was over one of them it will create a shadow copy of the ghost
    for(y = 0; y < MAZE_HEIGHT; y++)
        for(x = 0; x < MAZE_WIDTH; x++)
        {
            if(iFieldArray[y][x].iObject < PME_OBJECT_PELLET_POWER1) iFieldArray[y][x].iObject = PME_OBJECT_NULL;
            else
            {
                // Reset only the power pellet removing any possible ghost
                if((iFieldArray[y][x].iObject & PME_OBJECT_PELLET_POWER1) == PME_OBJECT_PELLET_POWER1) iFieldArray[y][x].iObject = PME_OBJECT_PELLET_POWER1;
                else if((iFieldArray[y][x].iObject & PME_OBJECT_PELLET_POWER2) == PME_OBJECT_PELLET_POWER2) iFieldArray[y][x].iObject = PME_OBJECT_PELLET_POWER2;
                else if((iFieldArray[y][x].iObject & PME_OBJECT_PELLET_POWER3) == PME_OBJECT_PELLET_POWER3) iFieldArray[y][x].iObject = PME_OBJECT_PELLET_POWER3;
                else if((iFieldArray[y][x].iObject & PME_OBJECT_PELLET_POWER4) == PME_OBJECT_PELLET_POWER4) iFieldArray[y][x].iObject = PME_OBJECT_PELLET_POWER4;
            }
        }

    // Fixed start position
    iFieldArray[23][14].iObject = PME_OBJECT_PACMAN;
    iFieldArray[14][12].iObject = PME_OBJECT_GHOST_RED;
    iFieldArray[14][13].iObject = PME_OBJECT_GHOST_PINK;
    iFieldArray[14][14].iObject = PME_OBJECT_GHOST_BLUE;
    iFieldArray[14][15].iObject = PME_OBJECT_GHOST_ORANGE;

    // Init the objects
    for(i = 0; i < vObjects.size(); ++i) vObjects[i]->msgGoInit();

    // Init the wave mode
    sGlobalWave.reset();

    // Workbench mode disables scattering wave
    if(pGlobalStatus->iGameType == PME_GAME_WORKBENCH) sGlobalWave.iChanges = PME_CRUISE_MODE;
    return 0;
}

// Validate the maze position as walkable or try to find a valid position around the original request up to a radius of 4
// It should cover all possible cases
Sint32 GameField::validateMazePosition(Sint32& iX, Sint32& iY)
{
    Sint32 iRadius = 1, iOX, iOY;

    // Quick "clipping"
    if(iX < 0) iX = 0;
    else if(iX > (MAZE_WIDTH - 1)) iX = MAZE_WIDTH - 1;
    if(iY < 0) iY = 0;
    else if(iY >(MAZE_HEIGHT - 1)) iY = MAZE_HEIGHT - 1;
    
    // Check with iRadius[1,4]
    iOX = iX;
    iOY = iY;
    while(iRadius < 4)
    {
        for(int y = -iRadius; y <= iRadius; y++)
        {
            for(int x = -iRadius; x <= iRadius; x++)
            {
                if(x*x + y * y <= iRadius * iRadius)
                {
                    iX = iOX + x;
                    iY = iOY + y;
                    if(getState(iX, iY) == PME_STATE_WALKABLE)
                    {
                        //printf("Radius %d (%d,%d)\n", iRadius, iX, iY);
                        return 0;
                    }
                }
            }
        }
        ++iRadius;
    }

    return -1;
}

// Return in the provided vector the posible movement options.
// At least always two points (maze has not dead-ends) and more than two, it is an intersection
Sint32 GameField::getMovementOptions(Sint32 iX, Sint32 iY, vector<MazePoint>& vMP)
{
    MazePoint vPoint;

    // Clear vector
    vMP.clear();
    
    // First priority: Up
    if(getState(iX, iY - 1) == PME_STATE_WALKABLE)
    {
        vPoint.iX = iX;
        vPoint.iY = iY - 1;
        vMP.push_back(vPoint);
    }       

    // Second priority: Left
    if(getState(iX - 1, iY) == PME_STATE_WALKABLE)
    {
        vPoint.iX = iX - 1;
        vPoint.iY = iY;
        vMP.push_back(vPoint);
    }
    // Left tunnel detected, allowit!
    else if((iX - 1) < 0)
    {
        vPoint.iX = iX - 1;
        vPoint.iY = iY;
        vMP.push_back(vPoint);
    }

    // Third priority: Down
    if(getState(iX, iY + 1) == PME_STATE_WALKABLE)
    {
        vPoint.iX = iX;
        vPoint.iY = iY + 1;
        vMP.push_back(vPoint);
    }
    
    // Last priority: Right
    if(getState(iX + 1, iY) == PME_STATE_WALKABLE)
    {
        vPoint.iX = iX + 1;
        vPoint.iY = iY;
        vMP.push_back(vPoint);
    }
    // Right tunnel detected, allowit!
    else if((iX + 1) > (MAZE_WIDTH - 1))
    {
        vPoint.iX = iX + 1;
        vPoint.iY = iY;
        vMP.push_back(vPoint);
    }

    return 0;
}

// Return current wave mode
Sint32 GameField::getWaveMode()
{
    return sGlobalWave.iMode;
}

// Set the wave mode. Only works for the two evading modes set by PacMan
Sint32 GameField::setWaveMode(Sint32 iMode)
{
    if(iMode == PME_GLOBAL_WAVE_EVADING)
    {
        sGlobalWave.iPreviousMode = sGlobalWave.iMode;
        sGlobalWave.iMode = PME_GLOBAL_WAVE_EVADING;
    }
    else if(iMode == PME_GLOBAL_WAVE_EVADING_END) sGlobalWave.iMode = PME_GLOBAL_WAVE_EVADING_END;    
    return 0;
}

// Restore previous wave mode
Sint32 GameField::restoreWaveMode()
{
    sGlobalWave.iMode = sGlobalWave.iPreviousMode;
    return 0;
}

// Update waves controlling the changes between them
Sint32 GameField::sGlobalWave::update()
{
    // Entry point
    if(iMode == 0)
    {
        iMode = PME_GLOBAL_WAVE_CHASING;
        iTicks = static_cast<Sint32>((PME_GHOST_CHASING_TIME * Main::Instance().ITimer().getLFR()) / 1000);
        return 0;
    }

    // Global evading for Ghosts
    if((iMode == PME_GLOBAL_WAVE_EVADING) || (iMode == PME_GLOBAL_WAVE_EVADING_END)) return 0;
    
    // After this number of changes, we fix chasing mode
    if(iChanges >= PME_CRUISE_MODE)
    {
        iMode = PME_GLOBAL_WAVE_CHASING;
        iTicks = static_cast<Sint32>((PME_GHOST_CHASING_TIME * Main::Instance().ITimer().getLFR()) / 1000);
    }

    // Keep changing states
    else if(iTicks < 0)
    {
        if(iMode == PME_GLOBAL_WAVE_CHASING)
        {
            iMode = PME_GLOBAL_WAVE_SCATTERING;
            ++iChanges;
            iTicks = static_cast<Sint32>((PME_GHOST_SCATTERING_TIME * Main::Instance().ITimer().getLFR()) / 1000);
        }
        else
        {
            iMode = PME_GLOBAL_WAVE_CHASING;
            ++iChanges;
            iTicks = static_cast<Sint32>((PME_GHOST_CHASING_TIME * Main::Instance().ITimer().getLFR()) / 1000);
        }  
    }   
    return 0;
}

// Reset the wave ticks mode
Sint32 GameField::sGlobalWave::reset()
{
    iMode = iChanges = iPreviousMode = 0;
    return 0;
}

// Show the message at the beginning of the maze
Sint32 GameField::messageStart()
{
    Uint64 iTime = 0, iTmp, iStart, iDelay;
    Uint8 iDone;
    SDL_Event eEvent;
    string sMazeNumber;
    Main& mC64 = Main::Instance();

    iSpecialTextCounter = 0;   
    sSpecialMessage = "Starting maze ";
    mC64.ITool().intToStrDec(iMazeReady, sMazeNumber);
    sSpecialMessage += sMazeNumber;
    iDone = 3; // Number of seconds 
    iDelay = (iDone * 1000);
    iRenderGraphicsStatus = RENDERGRAPHICS_START;
    mC64.IAudioTrackMgr().get(ResourceManager::Instance().get(RM_SND_GAMESTARTING))->play();
    iStart = mC64.ITimer().getTicksNow();

    while(iDone > 0)
    {
        // Logic update
        iTmp = mC64.ITimer().getTicksNow();
        if(iTime == 0) 
        {
            iTime = iTmp;			
            mC64.ITool().intToStrDec(iDone, sSpecialMessageCounter);
        }
        else if(iTmp > (iTime + 1000)) 
        { 
            iTime = 0; 
            iDone--;
        }
        iSpecialTextCounter += 2;
        // Try to avoid too much error propagation on the delay (it will never be equal to iDelayStep)
        // As soon as the total time spent here is greater than the delay, we return back
        if(iTmp > (iStart + iDelay)) iDone = 0;

        // Event loop for allowing the rendering to kick-in
        while(mC64.update(&eEvent));
    }
    // Clear it as we will be using for get ready message
    sSpecialMessage.clear();
    return PME_LOOP;
}

// Show a message at the end of game
// It receives as parameter: PME_BREAK(game aborted) or PME_GAME_OVER(game over)
Sint32 GameField::messageEnd(Sint32 bFlag)
{
    Sint32 iDone = PME_LOOP;
    SDL_Event eEvent;
    string sText;
    Main& mC64 = Main::Instance();
    Panel* myPanel = mC64.IGUIMgr().getPanel(ResourceManager::Instance().get(RM_PANEL_GAME));

    // Fade out music and play special game end music
    mC64.IAudioTrackMgr().fadeOutTag(ATT_MUSIC, 500);
    // Select mode
    switch(bFlag)
    {
    case PME_BREAK:
        myPanel->getWidget(ID_GAME_LABEL)->setText("Game Aborted");
        mC64.IAudioTrackMgr().get(ResourceManager::Instance().get(RM_SND_GAMEABORT))->play();
        break;
    case PME_GAME_OVER:
        myPanel->getWidget(ID_GAME_LABEL)->setText(" Game Over! ");
        mC64.IAudioTrackMgr().get(ResourceManager::Instance().get(RM_SND_GAMEOVER))->play();
        break;
    default:
        return -1;
    }

    // If not enought points for entering in HoF, disable input name widget
    if(pGlobalStatus->iLowestScore > pGlobalStatus->iPoints) myPanel->getWidget(ID_GAME_ENTERNAME)->hide();
    else myPanel->getWidget(ID_GAME_ENTERNAME)->show();

    // End maze message loop
    mC64.ICursorMgr().show();
    mC64.IGUIMgr().getPanel(ResourceManager::Instance().get(RM_PANEL_GAME))->baseWidget().show();
    while(iDone == PME_LOOP)
    {
        // Logic update
         
        // Event loop
        while(mC64.update(&eEvent))
        {
            switch(eEvent.type)
            {
            case C64_EVENT:
                // Check for widget activity
                if(eEvent.user.code == C64_EVENT_WIDGET)
                {
                    if(*static_cast<Sint32*>(eEvent.user.data1) == ID_GAME_CLOSE)
                    {
                        mC64.IAudioTrackMgr().get(ResourceManager::Instance().get(RM_SND_CLICKOK))->play();
                        iDone = PME_BREAK;
                        break;
                    }
                }
                break;
            }
        }
    }

    // Get the name of the player. Default to PacMan
    myPanel->getWidget(ID_GAME_ENTERNAME)->getText(sText);
    if(sText.empty()) sText = "PacMan";
    mC64.ITool().szCopy(pGlobalStatus->szName, sText.c_str(), sizeof(pGlobalStatus->szName));
    #ifdef DEBUG_INTERNAL
    mC64.ILogMgr().get()->msg(LML_INFO, " Player '%s' - Points '%d' (%d<->%d)\n", sText.c_str(), pGlobalStatus->iPoints, pGlobalStatus->iLowestScore, pGlobalStatus->iHighestScore);
    #endif

    // Return
    sSpecialMessage.clear();
    mC64.ICursorMgr().hide();
    mC64.IGUIMgr().getPanel(ResourceManager::Instance().get(RM_PANEL_GAME))->baseWidget().hide();
    return PME_LOOP;
}


// Render a special text effect
Sint32 GameField::specialText(Font* pFont, Sint32 iX, Sint32 iY, string &sText, Sint32 iCount)
{
    double dAngle = 0;
    Sint32 i, iXPos, iYPos;
    char sChar[2];

    // Count modulates the effect
    dAngle = (double)iCount * 5.0;

    // One iteration per each character on the string
    for(i = 0; i < (Sint32)sText.length(); i++)
    {
        sChar[0] = sText[i];
        sChar[1] = '\0';
        if(i == 0) iXPos = iX;
        iYPos = (Sint32)(iY + sin(SDL_PI_D / 180 * (dAngle + i * 12)) * pFont->getHeight());
        pFont->setPosition((float)iXPos, (float)iYPos);
        pFont->render(sChar);
        iXPos = iXPos + pFont->getWidth(sChar);
    }
    return 0;
}

// Render debug information
Sint32 GameField::debug()
{   
    Sint32 x, y, iFW, iFH;
    string sDebug, sTmp;
    Main& mC64 = Main::Instance();
    Font* pFont = mC64.IFontMgr().get(ResourceManager::Instance().get(RM_FONT_CONSOLE));

    
    if(iMazeReady == 0) return -1;
    iFH = pFont->getHeight();

    // Render each tile attributes on screen
    for(y = 0; y < MAZE_HEIGHT; y++)
    {
        for(x = 0; x < MAZE_WIDTH; x++)
        {
            // Select debug mode. No room for showing all
            switch(iDebugMode)
            {
                // Tile coordinates
            case 1:
                mC64.ITool().intToStrDec(x, sTmp);
                sDebug = sTmp; sDebug += ",";
                mC64.ITool().intToStrDec(y, sTmp);
                sDebug += sTmp;
                break;
                        
                // Tile state
            case 2:
                mC64.ITool().intToStrDec(iFieldArray[y][x].iState, sTmp);
                sDebug = sTmp;
                break;

                // Tile item
            case 3:
                mC64.ITool().intToStrDec(iFieldArray[y][x].iItem, sTmp);
                sDebug = sTmp;
                break;

                // Tile object
            case 4:
                mC64.ITool().intToStrDec(iFieldArray[y][x].iObject, sTmp);
                sDebug = sTmp;
                break;
            }

            // Center debug info in the tile
            iFW = pFont->getWidth(sDebug);
            pFont->setPosition((float)((x * MAZE_TILE_SIZE + iMazePixelX) - ((iFW - MAZE_TILE_SIZE) / 2)), (float)((y * MAZE_TILE_SIZE + iMazePixelY) - ((iFH - MAZE_TILE_SIZE) / 2)));
            pFont->render(sDebug);
        }
    }    

    // Display debug mode info
    sDebug = "Debug mode: ";
    switch(iDebugMode)
    {
    case 0:
        sDebug += "info";
        break;
    case 1:
        sDebug += "coordinates";
        break;
    case 2:
        sDebug += "states";
        break;
    case 3:
        sDebug += "items";
        break;
    case 4:
        sDebug += "objects";
        break;
    }
    pFont->setPosition((float)PME_DEBUG_PANEL_GAMEFIELD_X, (float)(PME_DEBUG_PANEL_GAMEFIELD_Y + iMazePixelY));
    pFont->render(sDebug);
    if(bDebugDisableObjectRender == 1) sDebug = "Objects rendering disabled";
    else sDebug = "Objects rendering enabled";
    pFont->setPosition((float)PME_DEBUG_PANEL_GAMEFIELD_X, (float)(PME_DEBUG_PANEL_GAMEFIELD_Y + iMazePixelY + 16));
    pFont->render(sDebug);
    sDebug = "Maze: ";   
    mC64.ITool().intToStrDec(iMazeReady, sTmp);
    sDebug += sTmp; sDebug += " - Wave: ";
    mC64.ITool().intToStrDec(sGlobalWave.iTicks, sTmp);
    sDebug += sTmp;
    pFont->setPosition((float)PME_DEBUG_PANEL_GAMEFIELD_X, (float)(PME_DEBUG_PANEL_GAMEFIELD_Y + iMazePixelY + 48));
    pFont->render(sDebug);
    sDebug = "Pellets: ";
    mC64.ITool().intToStrDec(iNumEatenPellets, sTmp);
    sDebug += sTmp; sDebug += "/";
    mC64.ITool().intToStrDec(iNumPellets, sTmp);
    sDebug += sTmp; sDebug += " (";
    mC64.ITool().intToStrDec(getEatenPelletsPercent(), sTmp);
    sDebug += sTmp; sDebug += "%)";
    pFont->setPosition((float)PME_DEBUG_PANEL_GAMEFIELD_X, (float)(PME_DEBUG_PANEL_GAMEFIELD_Y + iMazePixelY + 64));
    pFont->render(sDebug);

    sDebug = "LFR/RFR: ";
    mC64.ITool().intToStrDec(mC64.ITimer().getCurrentLFR(), sTmp);
    sDebug += sTmp; sDebug += "/";
    mC64.ITool().intToStrDec(mC64.ITimer().getCurrentRFR(), sTmp);
    sDebug += sTmp;
    pFont->setPosition((float)PME_DEBUG_PANEL_GAMEFIELD_X, (float)(PME_DEBUG_PANEL_GAMEFIELD_Y + iMazePixelY + 80));
    pFont->render(sDebug);

    return 0;
}

// Add the points to pacman score
Sint32 GameField::addPoints(Sint32 iP)
{
    pGlobalStatus->iPoints += iP;
    if(pGlobalStatus->iHighestScore < pGlobalStatus->iPoints) pGlobalStatus->iHighestScore = pGlobalStatus->iPoints;
    return 0;
}

// Return the index position on vObjects of the requested object ID
// Althought the first position belongs to PacMan and the next 4 to the ghosts, instead of using this trick
// we prefer to be on the safe side and look for the position.
Sint32 GameField::getObjectIndex(Sint32 iID)
{
    Sint32 i;

    for(i = 0; i < vObjects.size(); ++i)
    {
        if(vObjects[i]->getID() == iID) return i;
    }
    return -1;
}

// Used by EVNTrainer for training brains
Actor* GameField::getActor(Sint32 iID)
{
    Sint32 i = getObjectIndex(iID);
    if(i != -1)
    {
        return reinterpret_cast<Actor*>(vObjects[i]);
    }
    return nullptr;
}

// Used by EVNTrainer for retrieving the maze number
Sint32 GameField::getMazeNumber()
{
    return iMazeReady;
}

// Template: dynamic 2D array of a given data type.
template <typename T> T** GameField::create2DArray(Sint32 height, Sint32 width)
{
    T **ppi;
    T *pool;
    T *curPtr;

    // Allocate memory for array of elements of column
    ppi = new(std::nothrow) T*[height];
    // Allocate memory for array of elements of each row
    pool = new(std::nothrow) T[width * height];

    // Now point the pointers in the right place
    curPtr = pool;
    for(int i = 0; i < height; i++)
    {
        *(ppi + i) = curPtr;
        curPtr += width;
    }
    // Return
    return ppi;
}

// Template: delete a dynamic 2D array.
template <typename T> Sint32 GameField::delete2DArray(T** Array)
{
    delete[] * Array;
    delete[] Array;

    return 0;
}
