/*----------------------------------------------------------------------
Pac-Man Evolution - Roberto Prieto
 Copyright (C) 2018-2025 MegaStorm Systems
contact@megastormsystems.com - http://www.megastormsystems.com

This software is provided 'as-is', without any express or implied
warranty.  In no event will the authors be held liable for any damages
arising from the use of this software.

Permission is granted to anyone to use this software for any purpose,
including commercial applications, and to alter it and redistribute it
freely, subject to the following restrictions:

1. The origin of this software must not be misrepresented; you must not
claim that you wrote the original software. If you use this software
in a product, an acknowledgment in the product documentation would be
appreciated but is not required.
2. Altered source versions must be plainly marked as such, and must not be
misrepresented as being the original software.
3. This notice may not be removed or altered from any source distribution.

------------------------------------------------------------------------

EVolution Neural Trainer

------------------------------------------------------------------------ */

#include "EVNTrainer.h"
#include "GeneticAlgorithm.h"
#include "GameField.h"
#include "BrainsFactory.h"
#include "Brains.h"
#include "Objects.h"
#include "ObjectsPacMan.h"
#include "Pac-Man_Evolution.h"
#include <algorithm> 

#define PME_EVNTRAINER_EVOLVED "Evolved"
#define PME_EVNTRAINER_TRAINING "Training-"

// Singleton stuff
EVNTrainer* EVNTrainer::mInstance = nullptr;

// Create a single instance. Initialize a random seed to our random number generators
EVNTrainer& EVNTrainer::Instance()
{
    if(!mInstance) mInstance = new(std::nothrow) EVNTrainer;
    return *mInstance;
}

// Explicit destructor.
void EVNTrainer::Terminate()
{
    if(mInstance) delete mInstance;
    mInstance = nullptr;
}

// Constructor
EVNTrainer::EVNTrainer()
{
    dCrossOverRate = PME_GA_CROSSOVER_RATE;
    dMutationRate = PME_GA_MUTATION_RATE;
    dMaxPerturbation = PME_GA_MAX_PERTURBATION;
    iNumElite = PME_GA_NUM_ELITE;
    iNumCopiesElite = PME_GA_NUM_COPIES_ELITE;
    iPopulation = PME_GA_POPULATION;
}

// Destructor
EVNTrainer::~EVNTrainer()
{
}

// Display information of the creature evolution status
Sint32 EVNTrainer::Creature::info()
{
    Log& mLog = *Main::Instance().ILogMgr().get();
    Sint32 i, k;

    if(iNumBestCreatures > 0)
    {
        for(k = 0; k < iNumBestCreatures && k < vBestCreaturesNN.size(); ++k)
        {
            mLog.msg(LML_INFO, "     BestCreature (%d/%d): ", k + 1, iNumBestCreatures);
            for(i = 0; i < vBestCreaturesNN[k].vWeights.size(); ++i)
            {
                mLog.msg(LML_INFO, "%.6f", vBestCreaturesNN[k].vWeights[i]);
                if(i != (vBestCreaturesNN[k].vWeights.size() - 1)) mLog.msg(LML_INFO, ",");                
            }
            mLog.msg(LML_INFO, "\n");
        }        
        mLog.msg(LML_INFO, "\n");
    }
    return 0;
}

// Creatures constructor
EVNTrainer::Creature::Creature(Sint32 iIdentifier)
{
    iID = iIdentifier;   
    iGeneration = 0;
    iBestCreatureGeneration = 0;
    dBestCreatureFitness = 0;
    iNumBestCreatures = 3; // Minimum must be 1!
    vBestCreaturesNN.clear();
    vBrains.clear();
    pGA = nullptr;
    vGenomesPopulation.clear();
}

// Creatures destructor
EVNTrainer::Creature::~Creature()
{
    if(pGA != nullptr) delete pGA;
    pGA = nullptr;
}

// Load a Ghost neural network into a ArtificialNeuralNet object
Sint32 EVNTrainer::load(string& sGhostName, ArtificialNeuralNet* pNN, string& sCDCFile, string& sXMLName)
{
    Sint32 idXML = -1;
    Main& mC64 = Main::Instance();
    Log& mLog = *Main::Instance().ILogMgr().get();
    XML* pXML;
    Sint32 iInputs, iOutputs, iLayers, iNpL, iGeneration, iActivationFunc;
    float fFitness;
    vector<double> vWeights;
    string sName, sType, sBuff;
    size_t sizePos;

    // With no sCDCFile use default one
    if(sCDCFile.empty()) sCDCFile = PME_GHOSTS_NN_FILE;
    // With no sXMLName use default one
    if(sXMLName.empty()) sXMLName = PME_GHOSTS_NN_BLOCK;

    // Extract name and type from ghost name
    sizePos = sGhostName.find_first_of("-", 0);
    if(sizePos == string::npos) return -1;
    sName = sGhostName.substr(0, sizePos);
    sType = sGhostName.substr(sizePos + 1, sGhostName.length());

    // First try to open CDC file and load the XML or failback to external XML
    if(mC64.ITool().fileExists(sCDCFile) == 0) idXML = mC64.IXMLMgr().load(sCDCFile, sXMLName);
    if(idXML < 0)
    {
        idXML = mC64.IXMLMgr().loadFromFile(sXMLName);
        if(idXML < 0)
        {
            mLog.msg(LML_NORMAL, "  [EVNTrainer] Warning: could not load XML file '%s' from '%s'\n", sXMLName.c_str(), sCDCFile.c_str());
            return -1;
        }
    }
    pXML = mC64.IXMLMgr().get(idXML);

    // Point to ghosts node
    if(pXML->nodePointTo(3, "Ghosts", sName.c_str(), sType.c_str()) != 0)
    {
        mLog.msg(LML_NORMAL, "  [EVNTrainer] Warning: could not find 'Ghosts\\%s\\%s' node\n", sName.c_str(), sType.c_str());
        mC64.IXMLMgr().close(idXML);
        return -1;
    }

    // Load attributes, any error will cause a failure
    if(pXML->getAttribute("generation", iGeneration) != 0)
    {
        mLog.msg(LML_NORMAL, "  [EVNTrainer] Warning: could not read 'generation' attribute for 'Ghosts\\%s\\%s' node\n", sName.c_str(), sType.c_str());
        mC64.IXMLMgr().close(idXML);
        return -1;
    }
    if(pXML->getAttribute("fitness", fFitness) != 0)
    {
        mLog.msg(LML_NORMAL, "  [EVNTrainer] Warning: could not read 'fitness' attribute for 'Ghosts\\%s\\%s' node\n", sName.c_str(), sType.c_str());
        mC64.IXMLMgr().close(idXML);
        return -1;
    }
    if(pXML->getAttribute("activation_func", iActivationFunc) != 0)
    {
        mLog.msg(LML_NORMAL, "  [EVNTrainer] Warning: could not read 'activation_func' attribute for 'Ghosts\\%s\\%s' node\n", sName.c_str(), sType.c_str());
        mC64.IXMLMgr().close(idXML);
        return -1;
    }
    if(iActivationFunc < 0) iActivationFunc = ANN_ACTIVATION_LINEAR;
    else if(iActivationFunc > 2) iActivationFunc = ANN_ACTIVATION_SIGMOID;
    if(pXML->getAttribute("inputs", iInputs) != 0)
    {
        mLog.msg(LML_NORMAL, "  [EVNTrainer] Warning: could not read 'inputs' attribute for 'Ghosts\\%s\\%s' node\n", sName.c_str(), sType.c_str());
        mC64.IXMLMgr().close(idXML);
        return -1;
    }
    if(pXML->getAttribute("outputs", iOutputs) != 0)
    {
        mLog.msg(LML_NORMAL, "  [EVNTrainer] Warning: could not read 'outputs' attribute for 'Ghosts\\%s\\%s' node\n", sName.c_str(), sType.c_str());
        mC64.IXMLMgr().close(idXML);
        return -1;
    }
    if(pXML->getAttribute("layers", iLayers) != 0)
    {
        mLog.msg(LML_NORMAL, "  [EVNTrainer] Warning: could not read 'layers' attribute for 'Ghosts\\%s\\%s' node\n", sName.c_str(), sType.c_str());
        mC64.IXMLMgr().close(idXML);
        return -1;
    }
    if(pXML->getAttribute("neurons_per_layer", iNpL) != 0)
    {
        mLog.msg(LML_NORMAL, "  [EVNTrainer] Warning: could not read 'neurons_per_layer' attribute for 'Ghosts\\%s\\%s' node\n", sName.c_str(), sType.c_str());
        mC64.IXMLMgr().close(idXML);
        return -1;
    }

    // Load neuron weights
    if(pXML->getText(sBuff) != 0)
    {
        mLog.msg(LML_NORMAL, "  [EVNTrainer] Warning: could not read neuron weights for 'Ghosts\\%s\\%s' node\n", sName.c_str(), sType.c_str());
        mC64.IXMLMgr().close(idXML);
        return -1;
    }

    // Close XML
    mC64.IXMLMgr().close(idXML);

    // Initialize neural network
    pNN->init(iInputs, iOutputs, iLayers, iNpL);
    pNN->setActivationFunction(iActivationFunc);
   
    // Load weights
    parseWeights((char*)sBuff.c_str(), vWeights);
    if(pNN->getNumberOfWeights() > vWeights.size())
    {
        mLog.msg(LML_NORMAL, "  [EVNTrainer] Warning: could not read neuron weights for 'Ghosts\\%s\\%s'. Parsed '%d' but needed '%d'\n", sName.c_str(), sType.c_str(), vWeights.size(), pNN->getNumberOfWeights());
        return -1;
    }
    pNN->setWeights(vWeights);

    // Print info
    #ifdef DEBUG_EVN
        mLog.msg(LML_INFO, "  [EVNTrainer] Info: '%s\\%s' ghost loaded: G('%d') - F('%.6f') - I('%d') - O('%d') - AF('%d')\n", 
            sName.c_str(), sType.c_str(), iGeneration, fFitness, iInputs, iOutputs, iActivationFunc);
    #endif
    
    return 0;
}

// Execute a training based on the globalstatus
Sint32 EVNTrainer::execute(GlobalStatus& pGlobalStatus, GameField& pGameField)
{
    Main& mC64 = Main::Instance();
    Log& mLog = *mC64.ILogMgr().get();
    Sint32 i, j, k, iDone, iLifes;    
   
    Actor* pActor;
    PacMan* pPacMan;
    
    float fFitness;
    string sName;

    // 1.Prepare Creatures vector
    if(initCreatures(pGlobalStatus) != 0)
    {
        mLog.msg(LML_NORMAL, "  [EVNTrainer] Warning: no ghost ready to be trained.\n");
        return -1;
    }

    // 2.Initialize Genetics algorithm for each creature
    for(k = 0; k < (Sint32)vCreatures.size(); ++k)
    {
        i = vCreatures[k]->vBrains[0]->getNeuralNet()->getNumberOfWeights();
        vCreatures[k]->pGA = new(std::nothrow) GeneticAlgorithm(iPopulation, dMutationRate, dCrossOverRate, dMaxPerturbation, iNumElite, iNumCopiesElite, i);
        for(j = 0; j < iPopulation; ++j)
        {
            vCreatures[k]->pGA->init(j, vCreatures[k]->vBrains[j]->getNeuralNet()->getWeights());
        }
        vCreatures[k]->vGenomesPopulation = vCreatures[k]->pGA->getChromos();
    }

    // 3.Main training loop: repeat the process iExecutions times    
    #ifdef DEBUG_EVN
        Sint32 iTimeStart, iTimeGenerationStart, iTimePopulationStart;
        mLog.msg(LML_INFO, "\n  [EVNTrainer] Starting evolution of '%d' creatures with a population of '%d' during '%d' generations\n", vCreatures.size(), iPopulation, pGlobalStatus.workBench.iExecutions);
        mLog.msg(LML_INFO, "               CrossOver rate of '%.2f' - Mutation rate of '%.2f' with a max perturbation of '%.2f'\n", PME_GA_CROSSOVER_RATE, PME_GA_MUTATION_RATE, PME_GA_MAX_PERTURBATION);
        FILE* fp = fopen(pGlobalStatus.workBench.szOutputCSV, "wt");
        fprintf(fp, "BestCreatureFitness,AveragePopulationFitness\n");
        iTimeStart = static_cast<Sint32>(mC64.ITimer().getTicksNow());
    #endif  
    i = 0;
    pGlobalStatus.iGameType = PME_GAME_WORKBENCH;
    while(i < pGlobalStatus.workBench.iExecutions)
    {
        // Next iteration
        ++i;

        #ifdef DEBUG_EVN
            mLog.msg(LML_INFO, "     Generation '%d':\n", i);
            iTimeGenerationStart = static_cast<Sint32>(mC64.ITimer().getTicksNow());
        #endif
        // Population loop. Allow multiples creatures training at the same time
        for(j = 0; j < iPopulation; ++j)
        {
            #ifdef DEBUG_EVN
                iTimePopulationStart = static_cast<Sint32>(mC64.ITimer().getTicksNow());
            #endif            
            // Initialize gamefield
            iDone = pGameField.init();
            pGlobalStatus.iPoints = 0;

            // Set training brains for our creatures
            for(k = 0; k < (Sint32)vCreatures.size(); ++k)
            {
                pActor = pGameField.getActor(vCreatures[k]->iID);
                if(pActor) pActor->setBrain((PME_BRAIN_TYPE_TRAINING0 << j) | vCreatures[k]->iID);
            }

            // Run the simulation
            while(iDone == PME_LOOP)
            {
                iDone = pGameField.execute();
                if(iDone == PME_MAZE_END) iDone = pGameField.nextMaze();
            }            

            // Assign fitness to our creatures based on the inverse of PacMan's points plus a modification based on the game end
            pPacMan = reinterpret_cast<PacMan*>(pGameField.getActor(PME_OBJECT_PACMAN));
            if(pPacMan != nullptr) iLifes = pPacMan->getLifes();
            else iLifes = PME_PACMAN_START_LIFES;
            if(iDone == PME_BREAK) // Aborted by the user or time ran out
            {
                pGlobalStatus.iPoints = pGlobalStatus.iPoints * (iLifes * 10);
            }
            fFitness = static_cast<float>(1.0 / (double)pGlobalStatus.iPoints);
            for(k = 0; k < (Sint32)vCreatures.size(); ++k) vCreatures[k]->vBrains[j]->setFitness(fFitness);

            // Output intermediate stats for each population
            #ifdef DEBUG_EVN
                mLog.msg(LML_INFO, "       Population '%d' reached Maze '%d' in %2.2f seconds with a fitness of '%.6f' (",
                    j, pGameField.getMazeNumber(), (float)(mC64.ITimer().getTicksNow() - iTimePopulationStart) / 1000.0f, fFitness);
                if(iDone == PME_BREAK) mLog.msg(LML_INFO, "Aborted, %d lifes)\n", iLifes);
                else  mLog.msg(LML_INFO, "PacMan died)\n");
            #endif  

            // Close the gamefield
            pGameField.close();
        }             
        
        // Apply genetics algorithm to our creatures
        for(k = 0; k < (Sint32)vCreatures.size(); ++k)
        {
            // Loop through the population
            for(j = 0; j < iPopulation; ++j)
            {
                // Get the fitness
                vCreatures[k]->vGenomesPopulation[j].fFitness = vCreatures[k]->vBrains[j]->getFitness();

                // Update stats for best population
                if(vCreatures[k]->vBrains[j]->getFitness() > vCreatures[k]->dBestCreatureFitness)
                {
                    vCreatures[k]->dBestCreatureFitness = vCreatures[k]->vBrains[j]->getFitness();
                    vCreatures[k]->iBestCreatureGeneration = vCreatures[k]->iGeneration;
                }

                // Gather best creatures genomes
                if(vCreatures[k]->vBestCreaturesNN.size() < vCreatures[k]->iNumBestCreatures)
                {
                    // First genomes and keep then sorted
                    vCreatures[k]->vBestCreaturesNN.push_back(vCreatures[k]->vGenomesPopulation[j]);
                    std::sort(vCreatures[k]->vBestCreaturesNN.begin(), vCreatures[k]->vBestCreaturesNN.end());
                }
                else if(vCreatures[k]->iNumBestCreatures > 0)
                {
                    // Compare with the lowest and if it is greater than it, add it!
                    if(vCreatures[k]->vGenomesPopulation[j].fFitness > vCreatures[k]->vBestCreaturesNN[0].fFitness)
                    {
                        vCreatures[k]->vBestCreaturesNN.push_back(vCreatures[k]->vGenomesPopulation[j]);
                    }
                }
            }

            // Keep only the best iNumBestCreatures
            if(vCreatures[k]->iNumBestCreatures > 0)
            {
                std::sort(vCreatures[k]->vBestCreaturesNN.rbegin(), vCreatures[k]->vBestCreaturesNN.rend());
                vCreatures[k]->vBestCreaturesNN.resize(vCreatures[k]->iNumBestCreatures);
                std::sort(vCreatures[k]->vBestCreaturesNN.begin(), vCreatures[k]->vBestCreaturesNN.end());
            }

            // Apply a new epoch. Avoid last execution for having rights creatures at saving time
            if(i < pGlobalStatus.workBench.iExecutions) 
                vCreatures[k]->vGenomesPopulation = vCreatures[k]->pGA->epoch(vCreatures[k]->vGenomesPopulation);            
            
            // Insert new evolved chromosomes(weights) in to the brains
            for(j = 0; j < iPopulation; ++j)
            {
                vCreatures[k]->vBrains[j]->getNeuralNet()->setWeights(vCreatures[k]->vGenomesPopulation[j].vWeights);
                //vCreatures[k]->vBrains[j]->setFitness(0); // Reset fitness
            }

            // Increase the generation
            ++vCreatures[k]->iGeneration;
        }            
       
        // Export to CSV best fitness and output intermediate stats for each generation 
        #ifdef DEBUG_EVN
            fprintf(fp, "%.6f,%.6f\n", vCreatures[0]->dBestCreatureFitness, vCreatures[0]->pGA->fitnessAverage());
            fflush(fp);
            mLog.msg(LML_INFO, "       Done in %2.2f seconds. BestF '%.6f' AverageF '%.6f' Mutations '%d' CrossOvers '%d' - BestF so far '%.6f'\n",
                (float)(mC64.ITimer().getTicksNow() - iTimeGenerationStart) / 1000.0f, vCreatures[0]->pGA->fitnessBest(), vCreatures[0]->pGA->fitnessAverage(),
                vCreatures[0]->pGA->getMutation(), vCreatures[0]->pGA->getCrossOver(), vCreatures[0]->dBestCreatureFitness);
        #endif                
    }

    // 4.Save
    saveCreatures(PME_GHOSTS_NN_FILE, PME_GHOSTS_NN_BLOCK);

    // 5. Creatures stats, remove them and return
    #ifdef DEBUG_EVN
        fclose(fp);
        mLog.msg(LML_INFO, "\n  [EVNTrainer] Evolution finished in %2.2f seconds. Best fitness '%.6f' reached on generation '%d'\n",
            (float)(mC64.ITimer().getTicksNow() - iTimeStart) / 1000.0f, vCreatures[0]->dBestCreatureFitness, vCreatures[0]->iBestCreatureGeneration + 1);
    #endif 
    for(k = 0; k < (Sint32)vCreatures.size(); ++k)
    {
        #ifdef DEBUG_EVN 
            vCreatures[k]->info();
        #endif
        delete vCreatures[k];
    }
    vCreatures.clear();
    return 0;
}

// Initialize creatures vector
Sint32 EVNTrainer::initCreatures(GlobalStatus& pGlobalStatus)
{
    Creature* pCreature;
    BrainEvolved* pBrain;
    Sint32 bProceed = 1;
    Sint32 i;

    // RedGhost
    if(pGlobalStatus.workBench.iGhostRedBrain >= PME_BRAIN_TYPE_TRAINING0)
    {
        // Assure the XML has the training set
        if(createDefaultANN("Red", PME_GHOSTS_NN_FILE, PME_GHOSTS_NN_BLOCK) == 0)
        {
            // Create a new creature for the the Red Ghost with iPopulation            
            pCreature = new(std::nothrow) Creature(PME_OBJECT_GHOST_RED);
            for(i = 0; i < iPopulation; ++i)
            {
                pBrain = reinterpret_cast<BrainEvolved*>(BrainsFactory::Instance().getBrain((PME_BRAIN_TYPE_TRAINING0 << i) | PME_OBJECT_GHOST_RED));
                pBrain->load();
                pCreature->vBrains.push_back(pBrain);
            }
            vCreatures.push_back(pCreature);
            bProceed = 0;
        }
    }

    // Pink Ghost
    if(pGlobalStatus.workBench.iGhostPinkBrain >= PME_BRAIN_TYPE_TRAINING0)
    {
        // Assure the XML has the training set
        if(createDefaultANN("Pink", PME_GHOSTS_NN_FILE, PME_GHOSTS_NN_BLOCK) == 0)
        {
            // Create a new creature with iPopulation            
            pCreature = new(std::nothrow) Creature(PME_OBJECT_GHOST_PINK);
            for(i = 0; i < iPopulation; ++i)
            {
                pBrain = reinterpret_cast<BrainEvolved*>(BrainsFactory::Instance().getBrain((PME_BRAIN_TYPE_TRAINING0 << i) | PME_OBJECT_GHOST_PINK));
                pBrain->load();
                pCreature->vBrains.push_back(pBrain);
            }
            vCreatures.push_back(pCreature);
            bProceed = 0;
        }
    }

    // Blue Ghost
    if(pGlobalStatus.workBench.iGhostBlueBrain >= PME_BRAIN_TYPE_TRAINING0)
    {
        // Assure the XML has the training set
        if(createDefaultANN("Blue", PME_GHOSTS_NN_FILE, PME_GHOSTS_NN_BLOCK) == 0)
        {
            // Create a new creature with iPopulation            
            pCreature = new(std::nothrow) Creature(PME_OBJECT_GHOST_BLUE);
            for(i = 0; i < iPopulation; ++i)
            {
                pBrain = reinterpret_cast<BrainEvolved*>(BrainsFactory::Instance().getBrain((PME_BRAIN_TYPE_TRAINING0 << i) | PME_OBJECT_GHOST_BLUE));
                pBrain->load();
                pCreature->vBrains.push_back(pBrain);
            }
            vCreatures.push_back(pCreature);
            bProceed = 0;
        }
    }

    // Orange Ghost
    if(pGlobalStatus.workBench.iGhostOrangeBrain >= PME_BRAIN_TYPE_TRAINING0)
    {
        // Assure the XML has the training set
        if(createDefaultANN("Orange", PME_GHOSTS_NN_FILE, PME_GHOSTS_NN_BLOCK) == 0)
        {
            // Create a new creature with iPopulation            
            pCreature = new(std::nothrow) Creature(PME_OBJECT_GHOST_ORANGE);
            for(i = 0; i < iPopulation; ++i)
            {
                pBrain = reinterpret_cast<BrainEvolved*>(BrainsFactory::Instance().getBrain((PME_BRAIN_TYPE_TRAINING0 << i) | PME_OBJECT_GHOST_ORANGE));
                pBrain->load();
                pCreature->vBrains.push_back(pBrain);
            }
            vCreatures.push_back(pCreature);
            bProceed = 0;
        }
    }

    // With at least one creature, we go ahead
    return bProceed;
}

// Create default ANN training set for a given ghost. 
// If any profile is missing, it will be created. The attributes (inputs, output, etc) are based
// on the Evolved profile and if this one is not available, we first create it.
Sint32 EVNTrainer::createDefaultANN(const string& sGhostName, string sCDCFile, string sXMLName)
{
    Sint32 idXML = -1;
    Main& mC64 = Main::Instance();
    Log& mLog = *Main::Instance().ILogMgr().get();
    XML* pXML;
    Sint32 iInputs, iOutputs, iLayers, iNpL, iNumWeights, iActivationFunc, i, j;
    vector<double> vWeights;
    string sName, sType, sBuff, sNumber;
    size_t sizePos;
    char szNumber[32];
    bool bCDCUsage = false, bSaveXML = false;

    // Extract name and type from ghost name
    sizePos = sGhostName.find_first_of("-", 0);
    if(sizePos == string::npos) sName = sGhostName;
    else sName = sGhostName.substr(0, sizePos);
    sType = PME_EVNTRAINER_EVOLVED;

    // First try to open CDC file and load the XML or failback to external XML
    if(mC64.ITool().fileExists(sCDCFile) == 0) idXML = mC64.IXMLMgr().load(sCDCFile, sXMLName);
    if(idXML < 0)
    {
        idXML = mC64.IXMLMgr().loadFromFile(sXMLName);
        if(idXML < 0)
        {
            idXML = mC64.IXMLMgr().create("Ghosts");
            if(idXML < 0)
            {
                mLog.msg(LML_NORMAL, "  [EVNTrainer] Warning: could not load XML file '%s' from '%s' and could not create a new one\n", sXMLName.c_str(), sCDCFile.c_str());
                return -1;
            }            
        }
    }
    else bCDCUsage = true;
    pXML = mC64.IXMLMgr().get(idXML);
        
    // Point to ghosts node
    if(pXML->nodePointTo(1, "Ghosts") != 0)
    {
        mLog.msg(LML_NORMAL, "  [EVNTrainer] Warning: 'Ghosts' node not found, creating it... ");
        if(pXML->nodeCreate("Ghosts") != 0)
        {            
            mLog.msg(LML_NORMAL, "error!\n");
            mC64.IXMLMgr().close(idXML);
            return -1;
        }
        mLog.msg(LML_NORMAL, "done!\n");
    }
    
    // Point to ghosts/name subnode
    if(pXML->nodePointTo(2, "Ghosts", sName.c_str()) != 0)
    {
        mLog.msg(LML_NORMAL, "  [EVNTrainer] Warning: 'Ghosts\\%s' subnode not found, creating it... ", sName.c_str());
        if(pXML->nodeCreate(sName) != 0)
        {
            mLog.msg(LML_NORMAL, "error!\n");
            mC64.IXMLMgr().close(idXML);
            return -1;
        }
        mLog.msg(LML_NORMAL, "done!\n");
    }

    // Point to ghosts/name/Evolved subnode
    if(pXML->nodePointTo(3, "Ghosts", sName.c_str(), sType.c_str()) != 0)
    {
        mLog.msg(LML_NORMAL, "  [EVNTrainer] Warning: 'Ghosts\\%s\\%s' subnode not found, creating it... ", sName.c_str(), sType.c_str());
        if(pXML->nodeCreate(sType) != 0)
        {
            mLog.msg(LML_NORMAL, "error!\n");
            mC64.IXMLMgr().close(idXML);
            return -1;
        }
        // Creating attributes
        iOutputs = PME_ANN_GHOST_OUTPUT;       
        if(sName == "Red")
        {
            iInputs = PME_ANN_GHOST_RED_INPUT;            
            iNpL = PME_ANN_GHOST_RED_HIDDEN;
            iActivationFunc = PME_ANN_GHOST_RED_ACTIVATION;
        }
        else if(sName == "Pink")
        {
            iInputs = PME_ANN_GHOST_PINK_INPUT;
            iNpL = PME_ANN_GHOST_PINK_HIDDEN;
            iActivationFunc = PME_ANN_GHOST_PINK_ACTIVATION;
        }
        else if(sName == "Blue")
        {
            iInputs = PME_ANN_GHOST_BLUE_INPUT;
            iNpL = PME_ANN_GHOST_BLUE_HIDDEN;
            iActivationFunc = PME_ANN_GHOST_BLUE_ACTIVATION;
        }
        else if(sName == "Orange")
        {
            iInputs = PME_ANN_GHOST_ORANGE_INPUT;
            iNpL = PME_ANN_GHOST_ORANGE_HIDDEN;
            iActivationFunc = PME_ANN_GHOST_ORANGE_ACTIVATION;
        }
        else
        {
            mLog.msg(LML_NORMAL, "  error! name is unsupported\n");
            mC64.IXMLMgr().close(idXML);
            return -1;
        }
        if(iNpL > 0)
        {
            iNumWeights = (iInputs * iNpL);
            iNumWeights = iNumWeights + (iNpL * iOutputs);
            #ifdef ANN_ENABLE_BIAS
                iNumWeights = iNumWeights + iNpL + iOutputs;
            #endif
            iLayers = 1;
        }
        else
        {
            iNumWeights = (iInputs * iOutputs);
            #ifdef ANN_ENABLE_BIAS
                iNumWeights = iNumWeights + iOutputs;
            #endif
            iLayers = 0;
        }
        pXML->setAttribute("generation", 0);
        pXML->setAttribute("fitness", 0.0f);
        pXML->setAttribute("activation_func", iActivationFunc);
        pXML->setAttribute("inputs", iInputs);
        pXML->setAttribute("outputs", iOutputs);
        pXML->setAttribute("layers", iLayers);
        pXML->setAttribute("neurons_per_layer", iNpL);

        // Save random weights
        sBuff.clear();        
        for(j = 0; j < iNumWeights; j++)
        {
            // Weight between ANN_WEIGHT_MIN and ANN_WEIGHT_MAX            
            snprintf(szNumber, sizeof(szNumber), "%.6f", generateWeights());
            sBuff += szNumber;
            if(j != (iNumWeights - 1)) sBuff += ",";
        }
        pXML->setText(sBuff);
        mLog.msg(LML_NORMAL, "done!\n");
    }

    // Load attributes of the Evolved profile for using the same attributes for the training ones. Any error will cause a failure   
    if(pXML->getAttribute("activation_func", iActivationFunc) != 0)
    {
        mLog.msg(LML_NORMAL, "  [EVNTrainer] Warning: could not read 'activation_func' attribute for 'Ghosts\\%s\\%s' node\n", sName.c_str(), sType.c_str());
        mC64.IXMLMgr().close(idXML);
        return -1;
    }
    if(iActivationFunc < 0) iActivationFunc = ANN_ACTIVATION_LINEAR;
    else if(iActivationFunc > 2) iActivationFunc = ANN_ACTIVATION_SIGMOID;
    if(pXML->getAttribute("inputs", iInputs) != 0)
    {
        mLog.msg(LML_NORMAL, "  [EVNTrainer] Warning: could not read 'inputs' attribute for 'Ghosts\\%s\\%s' node\n", sName.c_str(), sType.c_str());
        mC64.IXMLMgr().close(idXML);
        return -1;
    }
    if(pXML->getAttribute("outputs", iOutputs) != 0)
    {
        mLog.msg(LML_NORMAL, "  [EVNTrainer] Warning: could not read 'outputs' attribute for 'Ghosts\\%s\\%s' node\n", sName.c_str(), sType.c_str());
        mC64.IXMLMgr().close(idXML);
        return -1;
    }
    if(pXML->getAttribute("layers", iLayers) != 0)
    {
        mLog.msg(LML_NORMAL, "  [EVNTrainer] Warning: could not read 'layers' attribute for 'Ghosts\\%s\\%s' node\n", sName.c_str(), sType.c_str());
        mC64.IXMLMgr().close(idXML);
        return -1;
    }
    if(pXML->getAttribute("neurons_per_layer", iNpL) != 0)
    {
        mLog.msg(LML_NORMAL, "  [EVNTrainer] Warning: could not read 'neurons_per_layer' attribute for 'Ghosts\\%s\\%s' node\n", sName.c_str(), sType.c_str());
        mC64.IXMLMgr().close(idXML);
        return -1;
    }

    // Load neuron weights
    if(pXML->getText(sBuff) != 0)
    {
        mLog.msg(LML_NORMAL, "  [EVNTrainer] Warning: could not read neuron weights for 'Ghosts\\%s\\%s' node\n", sName.c_str(), sType.c_str());
        mC64.IXMLMgr().close(idXML);
        return -1;
    }

    // Load weights just for getting number of them
    iNumWeights = parseWeights((char*)sBuff.c_str(), vWeights);
    
    // Loop checking that the number of generations exist or create it
    for(i = 0; i < PME_GA_POPULATION; ++i)
    {
        sBuff = PME_EVNTRAINER_TRAINING;
        mC64.ITool().intToStrDec(i, sNumber);
        sBuff += sNumber;

        // Point to this ghost, if exists do nothing, otherwise create it
        if(pXML->nodePointTo(3, "Ghosts", sName.c_str(), sBuff.c_str()) != 0)
        {
            mLog.msg(LML_NORMAL, "  [EVNTrainer] Info: creating '%s' node. Weights in the interval of [%d,%d] with precision '%.6f'\n", sBuff.c_str(), ANN_WEIGHT_MIN, ANN_WEIGHT_MAX, ANN_PRECISION);
            pXML->nodePointTo(2, "Ghosts", sName.c_str());
            pXML->nodeCreate(sBuff);
            pXML->setAttribute("generation", 0);
            pXML->setAttribute("fitness", 0.0f);
            pXML->setAttribute("activation_func", iActivationFunc);
            pXML->setAttribute("inputs", iInputs);
            pXML->setAttribute("outputs", iOutputs);
            pXML->setAttribute("layers", iLayers);
            pXML->setAttribute("neurons_per_layer", iNpL);
            
            sBuff.clear();            
            for(j = 0; j < iNumWeights; j++)
            {
                // Weight between ANN_WEIGHT_MIN and ANN_WEIGHT_MAX            
                snprintf(szNumber, sizeof(szNumber), "%.6f", generateWeights());
                sBuff += szNumber;
                if(j != (iNumWeights - 1)) sBuff += ",";
            }
            pXML->setText(sBuff);
            bSaveXML = true;
        }
    }

    // Update the XML and close it
    if(bSaveXML)
    {
        if(bCDCUsage) pXML->save(sCDCFile);
        else pXML->saveToFile(sXMLName);       
    }
    mC64.IXMLMgr().close(idXML);

    return 0;
}

// Save creatures to XML
Sint32 EVNTrainer::saveCreatures(string sCDCFile, string sXMLName)
{
    Main& mC64 = Main::Instance();
    Log& mLog = *Main::Instance().ILogMgr().get();
    XML* pXML;    
    bool bCDCUsage = false;
    Sint32 idXML = -1, i, k;
    string sBuff, sNumber, sName;

    // With no sCDCFile use default one
    if(sCDCFile.empty()) sCDCFile = PME_GHOSTS_NN_FILE;
    // With no sXMLName use default one
    if(sXMLName.empty()) sXMLName = PME_GHOSTS_NN_BLOCK;

    // First try to open CDC file and load the XML or failback to external XML
    if(mC64.ITool().fileExists(sCDCFile) == 0) idXML = mC64.IXMLMgr().load(sCDCFile, sXMLName);
    if(idXML < 0)
    {
        idXML = mC64.IXMLMgr().loadFromFile(sXMLName);
        if(idXML < 0)
        {
            mLog.msg(LML_NORMAL, "  [EVNTrainer] Warning: could not load XML file '%s' from '%s'\n", sXMLName.c_str(), sCDCFile.c_str());
            return -1;
        }
    }
    else bCDCUsage = true;
    pXML = mC64.IXMLMgr().get(idXML);

    // Creatures loop
    for(k = 0; k < (Sint32)vCreatures.size(); ++k)
    {
        // Red Ghost
        if(vCreatures[k]->iID == PME_OBJECT_GHOST_RED) sName = "Red";
        // Pink Ghost
        else if(vCreatures[k]->iID == PME_OBJECT_GHOST_PINK) sName = "Pink";
        // Blue Ghost
        else if(vCreatures[k]->iID == PME_OBJECT_GHOST_BLUE) sName = "Blue";
        // Orange Ghost
        else if(vCreatures[k]->iID == PME_OBJECT_GHOST_ORANGE) sName = "Orange";
        // Error!
        else
        {
            mLog.msg(LML_NORMAL, "  [EVNTrainer] Warning: internal creature ID is unknown\n");
            mC64.IXMLMgr().close(idXML);
            return -1;
        }
        
        // Point to evolved node
        if(pXML->nodePointTo(3, "Ghosts", sName.c_str(), PME_EVNTRAINER_EVOLVED) != 0)
        {
            mLog.msg(LML_NORMAL, "  [EVNTrainer] Warning: could not find 'Ghosts\\%s\\%s' node\n", sName.c_str(), PME_EVNTRAINER_EVOLVED);
            mC64.IXMLMgr().close(idXML);
            return -1;
        }
        // Update this node
        updateNode(pXML, vCreatures[k], -1);

        // Point to training nodes
        for(i = 0; i < PME_GA_POPULATION; ++i)
        {
            sBuff = PME_EVNTRAINER_TRAINING;
            mC64.ITool().intToStrDec(i, sNumber);
            sBuff += sNumber;

            // Point to training node number i
            if(pXML->nodePointTo(3, "Ghosts", sName.c_str(), sBuff.c_str()) != 0)
            {
                mLog.msg(LML_NORMAL, "  [EVNTrainer] Warning: could not find 'Ghosts\\%s\\%s' node\n", sName.c_str(), sBuff.c_str());
                mC64.IXMLMgr().close(idXML);
                return -1;
            }
            // Update this node
            updateNode(pXML, vCreatures[k], i);
        }        
    }

    // Save and return
    if(bCDCUsage) pXML->save(sCDCFile);
    else pXML->saveToFile(sXMLName);
    mC64.IXMLMgr().close(idXML);
    return 0;
}

// Update current pXML node using pCreatures. iNum is set to the brain number or -1 for the best one
Sint32 EVNTrainer::updateNode(XML* pXML, Creature* pCreature, Sint32 iNum)
{
    Sint32 iGeneration, j;
    float fFitness;
    char szNumber[32];
    string sWeights;
    vector<double> pWeights;

    // Read current node attributes: generation and fitness
    if(pXML->getAttribute("generation", iGeneration) != 0) return -1;
    if(pXML->getAttribute("fitness", fFitness) != 0) return -1;

    // Read attributes from our creature
    if(iNum < 0) // Evolved node that keeps the best creature
    {
        // If the fitness is lower, do not modify this node
        if(fFitness > pCreature->dBestCreatureFitness) return 0;
        iGeneration = iGeneration + 1 + pCreature->iBestCreatureGeneration;
        fFitness = static_cast<float>(pCreature->dBestCreatureFitness); 
        pWeights = pCreature->vBestCreaturesNN[0].vWeights;
    }
    else // Training node
    {
        iGeneration += pCreature->iGeneration;
        fFitness = static_cast<float>(pCreature->vBrains[iNum]->getFitness());
        pWeights = pCreature->vBrains[iNum]->getNeuralNet()->getWeights();
    }

    // Save new attributes to current node
    pXML->setAttribute("generation", iGeneration);
    fFitness = applyWeightPrecision(fFitness);
    snprintf(szNumber, sizeof(szNumber), "%.6f", fFitness);
    sWeights = szNumber;
    pXML->setAttribute("fitness", sWeights);

    // Save new weights
    pXML->removeText();
    sWeights.clear();
    for(j = 0; j < pWeights.size(); j++)
    {        
        snprintf(szNumber, sizeof(szNumber), "%.6f", pWeights[j]);
        sWeights += szNumber;
        if(j != (pWeights.size() - 1)) sWeights += ",";
    }
    pXML->setText(sWeights);
    
    return 0;
}

// Parse from a char* of doubles to a vector<double>
Sint32 EVNTrainer::parseWeights(char* from, vector<double>& to)
{
    Sint32 iNum = 0;
    char  seps[] = ",\n\t";
    char *token;
    double dTmp;

    token = strtok(from, seps);
    while(token != NULL)
    {
        dTmp = atof(token);
        to.push_back(dTmp);
        token = strtok(NULL, seps);
        ++iNum;
    }
    return iNum;
}
