/*----------------------------------------------------------------------
Pac-Man Evolution - Roberto Prieto
 Copyright (C) 2018-2025 MegaStorm Systems
contact@megastormsystems.com - http://www.megastormsystems.com

This software is provided 'as-is', without any express or implied
warranty.  In no event will the authors be held liable for any damages
arising from the use of this software.

Permission is granted to anyone to use this software for any purpose,
including commercial applications, and to alter it and redistribute it
freely, subject to the following restrictions:

1. The origin of this software must not be misrepresented; you must not
claim that you wrote the original software. If you use this software
in a product, an acknowledgment in the product documentation would be
appreciated but is not required.
2. Altered source versions must be plainly marked as such, and must not be
misrepresented as being the original software.
3. This notice may not be removed or altered from any source distribution.

------------------------------------------------------------------------

Brains Factory class

------------------------------------------------------------------------ */

#include "BrainsFactory.h"
#include "Brains.h"
#include "GameField.h"
#include "EVNTrainer.h"

// Singleton stuff
BrainsFactory* BrainsFactory::mInstance = nullptr;

// Create a single instance
BrainsFactory& BrainsFactory::Instance()
{
    if(!mInstance) mInstance = new(std::nothrow) BrainsFactory;
    return *mInstance;
}

// Explicit destructor.
void BrainsFactory::Terminate()
{
    if(mInstance) delete mInstance;
    mInstance = nullptr;
}

// Constructor
BrainsFactory::BrainsFactory()
{
    Brain* pNewBrain;
    Sint32 i;

    // Create and register the brains. With error control
    mapBrains.clear();
    pNewBrain = new(std::nothrow) Brain();
    if(mapBrains.insert(std::make_pair(PME_BRAIN_TYPE_RANDOM, pNewBrain)).second == false) delete pNewBrain;
    
    pNewBrain = new(std::nothrow) BrainPacManHuman();
    if(mapBrains.insert(std::make_pair(PME_BRAIN_TYPE_HUMAN | PME_OBJECT_PACMAN, pNewBrain)).second == false) delete pNewBrain;

    pNewBrain = new(std::nothrow) BrainPacMan();
    if(mapBrains.insert(std::make_pair(PME_BRAIN_TYPE_FIXED | PME_OBJECT_PACMAN, pNewBrain)).second == false) delete pNewBrain;

    pNewBrain = new(std::nothrow) BrainRedGhost();
    if(mapBrains.insert(std::make_pair(PME_BRAIN_TYPE_FIXED | PME_OBJECT_GHOST_RED, pNewBrain)).second == false) delete pNewBrain;
    pNewBrain = new(std::nothrow) BrainEvolvedRedGhost(-1);
    if(mapBrains.insert(std::make_pair(PME_BRAIN_TYPE_EVOLVED | PME_OBJECT_GHOST_RED, pNewBrain)).second == false) delete pNewBrain;
    for(i = 0; i < PME_GA_POPULATION; ++i)
    {
        pNewBrain = new(std::nothrow) BrainEvolvedRedGhost(i);
        if(mapBrains.insert(std::make_pair((PME_BRAIN_TYPE_TRAINING0 << i) | PME_OBJECT_GHOST_RED, pNewBrain)).second == false) delete pNewBrain;
    }

    pNewBrain = new(std::nothrow) BrainPinkGhost();
    if(mapBrains.insert(std::make_pair(PME_BRAIN_TYPE_FIXED | PME_OBJECT_GHOST_PINK, pNewBrain)).second == false) delete pNewBrain;
    pNewBrain = new(std::nothrow) BrainEvolvedPinkGhost(-1);
    if(mapBrains.insert(std::make_pair(PME_BRAIN_TYPE_EVOLVED | PME_OBJECT_GHOST_PINK, pNewBrain)).second == false) delete pNewBrain;
    for(i = 0; i < PME_GA_POPULATION; ++i)
    {
        pNewBrain = new(std::nothrow) BrainEvolvedPinkGhost(i);
        if(mapBrains.insert(std::make_pair((PME_BRAIN_TYPE_TRAINING0 << i) | PME_OBJECT_GHOST_PINK, pNewBrain)).second == false) delete pNewBrain;
    }

    pNewBrain = new(std::nothrow) BrainBlueGhost();
    if(mapBrains.insert(std::make_pair(PME_BRAIN_TYPE_FIXED | PME_OBJECT_GHOST_BLUE, pNewBrain)).second == false) delete pNewBrain;
    pNewBrain = new(std::nothrow) BrainEvolvedBlueGhost(-1);
    if(mapBrains.insert(std::make_pair(PME_BRAIN_TYPE_EVOLVED | PME_OBJECT_GHOST_BLUE, pNewBrain)).second == false) delete pNewBrain;
    for(i = 0; i < PME_GA_POPULATION; ++i)
    {
        pNewBrain = new(std::nothrow) BrainEvolvedBlueGhost(i);
        if(mapBrains.insert(std::make_pair((PME_BRAIN_TYPE_TRAINING0 << i) | PME_OBJECT_GHOST_BLUE, pNewBrain)).second == false) delete pNewBrain;
    }

    pNewBrain = new(std::nothrow) BrainOrangeGhost();
    if(mapBrains.insert(std::make_pair(PME_BRAIN_TYPE_FIXED | PME_OBJECT_GHOST_ORANGE, pNewBrain)).second == false) delete pNewBrain;
    pNewBrain = new(std::nothrow) BrainEvolvedOrangeGhost(-1);
    if(mapBrains.insert(std::make_pair(PME_BRAIN_TYPE_EVOLVED | PME_OBJECT_GHOST_ORANGE, pNewBrain)).second == false) delete pNewBrain;
    for(i = 0; i < PME_GA_POPULATION; ++i)
    {
        pNewBrain = new(std::nothrow) BrainEvolvedOrangeGhost(i);
        if(mapBrains.insert(std::make_pair((PME_BRAIN_TYPE_TRAINING0 << i) | PME_OBJECT_GHOST_ORANGE, pNewBrain)).second == false) delete pNewBrain;
    }

    #ifdef DEBUG_INTERNAL
    Main::Instance().ILogMgr().get()->msg(LML_NORMAL, "  [BrainFactory] Info: registered '%d' brains.\n", mapBrains.size());
    #endif    
}

// Destructor
BrainsFactory::~BrainsFactory()
{
    // Destroy all brains
    for(auto itBrain = mapBrains.begin(); itBrain != mapBrains.end(); ++itBrain)
    {
        delete itBrain->second;
    }
    mapBrains.clear();
}

// Generic think method
Sint32 BrainsFactory::think(Actor* pActor, Sint32 iBrainID, Sint32& iTX, Sint32& iTY, GameField* pGF)
{
    auto itBrain = mapBrains.find(iBrainID);
    if(itBrain != mapBrains.end())
    {
        itBrain->second->think(pActor, iTX, iTY, pGF);
        return 0;
    }
    return -1;
}

// Validate that a given id corresponds to a valid brain
bool BrainsFactory::validate(Sint32 iBrainID)
{
    auto itBrain = mapBrains.find(iBrainID);
    if(itBrain != mapBrains.end()) return true;
    return false;
}

// Get name of a given brain id
Sint32 BrainsFactory::getName(Sint32 iBrainID, string& sN)
{
    auto itBrain = mapBrains.find(iBrainID);
    if(itBrain != mapBrains.end())
    {
        itBrain->second->getName(sN);
        return 0;
    }
    return -1;
}

// Get target type
Sint32 BrainsFactory::getTargetType(Sint32 iBrainID)
{
    auto itBrain = mapBrains.find(iBrainID);
    if(itBrain != mapBrains.end())
    {
        return itBrain->second->getTargetType();
    }
    return -1;
}

// Get a pointer to the brain
Brain* BrainsFactory::getBrain(Sint32 iBrainID)
{
    auto itBrain = mapBrains.find(iBrainID);
    if(itBrain != mapBrains.end())
    {
        return itBrain->second;
    }
    return nullptr;
}
