/*----------------------------------------------------------------------
Pac-Man Evolution - Roberto Prieto
 Copyright (C) 2018-2025 MegaStorm Systems
contact@megastormsystems.com - http://www.megastormsystems.com

This software is provided 'as-is', without any express or implied
warranty.  In no event will the authors be held liable for any damages
arising from the use of this software.

Permission is granted to anyone to use this software for any purpose,
including commercial applications, and to alter it and redistribute it
freely, subject to the following restrictions:

1. The origin of this software must not be misrepresented; you must not
claim that you wrote the original software. If you use this software
in a product, an acknowledgment in the product documentation would be
appreciated but is not required.
2. Altered source versions must be plainly marked as such, and must not be
misrepresented as being the original software.
3. This notice may not be removed or altered from any source distribution.

------------------------------------------------------------------------

Brains classes

------------------------------------------------------------------------ */

#include "Brains.h"
#include "BrainsFactory.h"
#include "ObjectsGhost.h"
#include "GameField.h"
#include "Pac-Man_Evolution.h"
#include "ArtificialNeuralNet.h"
#include "EVNTrainer.h"

// ---------- Base brain ----------
// Constructor
Brain::Brain()
{
    sName = "Random";
    iTargetType = PME_BRAIN_HIGHLEVEL_TARGET;
}

// Destructor
Brain::~Brain() {}

// Default think method return a random target position
Sint32 Brain::think(Actor* pActor, Sint32 &iTX, Sint32 &iTY, GameField* pGF)
{
    iTX = Main::Instance().ITool().randWELL() % MAZE_WIDTH;
    iTY = Main::Instance().ITool().randWELL() % MAZE_HEIGHT;
    return 0;
}

// Get brain name
void Brain::getName(string& sN)
{
    sN = sName;
}

// Get Target type
Sint32 Brain::getTargetType()
{
    return iTargetType;
}

// ---------- Evolved brain ----------
BrainEvolved::BrainEvolved() : Brain()
{
    sName = "Evolved";
    pNeuralNet = new(std::nothrow) ArtificialNeuralNet();
    iNNInput = 0;
    iNNOutput = 0;
    fFitness = 0.0f;
}

BrainEvolved::~BrainEvolved()
{
    if(pNeuralNet != nullptr) delete pNeuralNet;
    pNeuralNet = nullptr;
}

Sint32 BrainEvolved::setFitness(float fF)
{
    if(fF < 0) fF = 0;
    fFitness = fF;
    return 0;
}

float BrainEvolved::getFitness()
{
    return fFitness;
}

ArtificialNeuralNet* BrainEvolved::getNeuralNet()
{
    if(pNeuralNet) return pNeuralNet;
    else return nullptr;
}

// Load a Evolved neural net from the given file, it checks for the input/output
Sint32 BrainEvolved::load(string sCDCFile, string sXMLName)
{
    // Load the neural network, we are using an extra weight for the bias parameter
    if(EVNTrainer::Instance().load(sName, pNeuralNet, sCDCFile, sXMLName) != 0)
    {
        Main::Instance().ILogMgr().get()->msg(LML_NORMAL, "  [BrainEvolved] Warning: neural network for ghost '%s' could not be loaded\n", sName.c_str());
        delete pNeuralNet;
        pNeuralNet = nullptr;
    }

    // Check input/output of NN loaded, manually fixed to a specific amount. 
    // On think() method we use this layout so can not break it.
    else if((pNeuralNet->getNumInputs() != iNNInput) || (pNeuralNet->getNumOutputs() != iNNOutput))
    {
        Main::Instance().ILogMgr().get()->msg(LML_NORMAL, "  [BrainEvolved] Warning: neural network loaded for ghost '%s' does not match with expected inputs('%d')/outputs('%d')\n", sName.c_str(), iNNInput, iNNOutput);
        delete pNeuralNet;
        pNeuralNet = nullptr;
    }
    #ifdef DEBUG_EVOLVED
    if(pNeuralNet) Main::Instance().ILogMgr().get()->msg(LML_INFO, "  [BrainEvolved] Info: neural network loaded for ghost '%s' with '%d' inputs, '%d' outputs and '%d' weights\n",
        sName.c_str(), pNeuralNet->getNumInputs(), pNeuralNet->getNumOutputs(), pNeuralNet->getNumberOfWeights());
    #endif

    return 0;
}

// ---------- PacMan brains ----------
// Human brain
BrainPacManHuman::BrainPacManHuman() : Brain() { sName = "PacMan-Human"; iTargetType = PME_BRAIN_IMMEDIATE_TARGET; }
Sint32 BrainPacManHuman::think(Actor* pActor, Sint32& iTX, Sint32& iTY, GameField* pGF) // Get human keyboard action and keep moving while possible
{
    Main& mC64 = Main::Instance();
    Sint32 iTmpX, iTmpY, iDirX, iDirY;

    // Get current PacMan position
    pGF->getObjectPosition(PME_OBJECT_PACMAN, iTX, iTY);
    iTmpX = iTX;
    iTmpY = iTY;

    // Modify target following the keystrokes
    if(mC64.getKeyState(SDLK_DOWN))
    {
        if(pGF->getState(iTX, iTY + 1) == PME_STATE_WALKABLE) ++iTY;
    }
    if(mC64.getKeyState(SDLK_UP))
    {
        if(pGF->getState(iTX, iTY - 1) == PME_STATE_WALKABLE) --iTY;
    }
    if(mC64.getKeyState(SDLK_RIGHT))
    {
        if((iTX + 1) > (MAZE_WIDTH - 1)) ++iTX;
        else if(pGF->getState(iTX + 1, iTY) == PME_STATE_WALKABLE) ++iTX;
    }
    if(mC64.getKeyState(SDLK_LEFT))
    {
        if((iTX - 1) < 0) --iTX;
        else if(pGF->getState(iTX - 1, iTY) == PME_STATE_WALKABLE) --iTX;        
    }

    // With no human action, keep moving into the same direction
    if(iTmpX == iTX && iTmpY == iTY)
    {
            // Get direction
            pGF->getObjectDirection(PME_OBJECT_PACMAN, iDirX, iDirY);

            // Apply direction
            iTmpX = iTmpX + 1 * iDirX;
            iTmpY = iTmpY + 1 * iDirY;
            if((iTmpX) > (MAZE_WIDTH - 1)) iTX = iTmpX;
            else if((iTmpX) < 0) iTX = iTmpX;
            else if(pGF->getState(iTmpX, iTmpY) == PME_STATE_WALKABLE)
            {
                iTX = iTmpX;
                iTY = iTmpY;
            }
    }

    return 0;
}
// Fixed brain
BrainPacMan::BrainPacMan() : Brain() { sName = "PacMan-Fixed";  }
Sint32 BrainPacMan::think(Actor* pActor,Sint32& iTX, Sint32& iTY, GameField* pGF) // Fixed rules I
{
    vector<MazePoint> vGhosts;
    MazePoint pointGhost;
    string sStateName;
    Sint32 i, iSelected = 0;
    double dLowest = 1000.0f;

    // Get current position
    pGF->getObjectPosition(PME_OBJECT_PACMAN, iTX, iTY);

    // Get all ghost positions (non-death and not at home), calculate the distance and get the closest one
    if(pGF->getObjectPosition(PME_OBJECT_GHOST_RED, pointGhost.iX, pointGhost.iY) == 0)
    {
        pGF->getObjectStateName(PME_OBJECT_GHOST_RED, sStateName);
        if(sStateName != "Death" && sStateName != "Init")
        {
            pointGhost.iReserved = PME_OBJECT_GHOST_RED;
            vGhosts.push_back(pointGhost);
        }
    }
    if(pGF->getObjectPosition(PME_OBJECT_GHOST_PINK, pointGhost.iX, pointGhost.iY) == 0)
    {
        pGF->getObjectStateName(PME_OBJECT_GHOST_PINK, sStateName);
        if(sStateName != "Death" && sStateName != "Init")
        {
            pointGhost.iReserved = PME_OBJECT_GHOST_PINK;
            vGhosts.push_back(pointGhost);
        }
    }
    if(pGF->getObjectPosition(PME_OBJECT_GHOST_BLUE, pointGhost.iX, pointGhost.iY) == 0)
    {
        pGF->getObjectStateName(PME_OBJECT_GHOST_BLUE, sStateName);
        if(sStateName != "Death" && sStateName != "Init")
        {
            pointGhost.iReserved = PME_OBJECT_GHOST_BLUE;
            vGhosts.push_back(pointGhost);
        }
    }
    if(pGF->getObjectPosition(PME_OBJECT_GHOST_ORANGE, pointGhost.iX, pointGhost.iY) == 0)
    {
        pGF->getObjectStateName(PME_OBJECT_GHOST_ORANGE, sStateName);
        if(sStateName != "Death" && sStateName != "Init")
        {
            pointGhost.iReserved = PME_OBJECT_GHOST_ORANGE;
            vGhosts.push_back(pointGhost);
        }
    }

    // Calculate the distance of all found ghosts to PacMan
    for(i = 0; i < vGhosts.size(); ++i) pActor->euclideanDistance(iTX, iTY, vGhosts[i]);

    // Get the lowest distance
    for(i = 0; i < vGhosts.size(); ++i)
    {
        if(dLowest > vGhosts[i].dDistance)
        {
            dLowest = vGhosts[i].dDistance;
            iSelected = i;
        }
    }

    // With no ghosts (debugging for example), go to nearest pellet
    if(vGhosts.size() == 0)
    {
        pGF->getClosestPellet(PME_OBJECT_PACMAN, iTX, iTY);
    }

    // When Wave mode is evading and we are not very far, go to closest ghost
    else if((pGF->getWaveMode() == PME_GLOBAL_WAVE_EVADING) && (vGhosts[iSelected].dDistance < 10))
    {
        iTX = vGhosts[iSelected].iX;
        iTY = vGhosts[iSelected].iY;
    }
    // When ghosts are far from PacMan, go to nearest pellet
    else if(vGhosts[iSelected].dDistance > 6.0)
    {
        pGF->getClosestPellet(PME_OBJECT_PACMAN, iTX, iTY);
    }
    // When ghosts are close to PacMan, try to evade
    else
    {
        iTX = vGhosts[iSelected].iX + 2 * (iTX - vGhosts[iSelected].iX);
        iTY = vGhosts[iSelected].iY + 2 * (iTY - vGhosts[iSelected].iY);
    }

    return 0;
}

// Red Ghost brains
// Fixed brain
BrainRedGhost::BrainRedGhost() : Brain() { sName = "Red-Fixed"; }
Sint32 BrainRedGhost::think(Actor* pActor, Sint32& iTX, Sint32& iTY, GameField* pGF) // Target PacMan position
{
    return pGF->getObjectPosition(PME_OBJECT_PACMAN, iTX, iTY);
}
// Evolved brain
BrainEvolvedRedGhost::BrainEvolvedRedGhost(Sint32 iID) : BrainEvolved()
{ 
    string sNumber;
    
    // Out of range
    if(iID > PME_GA_POPULATION)
    {
        sName = "Red-Unknown";
        return;
    }
    iNNInput = PME_ANN_GHOST_RED_INPUT;
    iNNOutput = PME_ANN_GHOST_OUTPUT;

    // Evolved ID
    if(iID < 0)
    {
        sName = "Red-" + sName;
        load();
    }
    // Training ID
    else
    {
        Main::Instance().ITool().intToStrDec(iID, sNumber);
        sName = "Red-Training";
        sName = sName + "-" + sNumber;
    }
}
Sint32 BrainEvolvedRedGhost::think(Actor* pActor, Sint32& iTX, Sint32& iTY, GameField* pGF)
{
    double dTmp;
    vector<double> vInputs, vOutputs;

    // In case of problems with the neural network, fallback to fixed target
    if(pNeuralNet == nullptr) return 0;

    // Get PacMan position
    pGF->getObjectPosition(PME_OBJECT_PACMAN, iTX, iTY);

    // Prepare inputs
    // 2 input: PacMan (x,y) -> normalization to [0,1]
    vInputs.clear();
    dTmp = ((double)iTX / (double)MAZE_WIDTH);
    vInputs.push_back(dTmp);
    dTmp = ((double)iTY / (double)MAZE_HEIGHT);
    vInputs.push_back(dTmp);

    // Call neural network
    vOutputs = pNeuralNet->update(vInputs);

    // Prepare outputs
    // 2 output from [0,1], convert it to target coordinates    
    if(!vOutputs.empty())
    {
        iTX = (Sint32)Main::Instance().ITool().round(vOutputs[0] * MAZE_WIDTH);
        iTY = (Sint32)Main::Instance().ITool().round(vOutputs[1] * MAZE_HEIGHT);
    }

    return 0;
}

// Pink Ghost brains
// Fixed brain
BrainPinkGhost::BrainPinkGhost() : Brain() { sName = "PinkGhost-Fixed"; }
Sint32 BrainPinkGhost::think(Actor* pActor, Sint32& iTX, Sint32& iTY, GameField* pGF) // Target PacMan position plus 4 maze points where he is facing to
{
    Sint32 iDirX, iDirY;
    
    // Get current position and direction
    pGF->getObjectPosition(PME_OBJECT_PACMAN, iTX, iTY);
    pGF->getObjectDirection(PME_OBJECT_PACMAN, iDirX, iDirY);
    
    // Calculate target
    iTX = iTX + 4 * iDirX;
    iTY = iTY + 4 * iDirY;
    
    return 0;
}
// Evolved brain
BrainEvolvedPinkGhost::BrainEvolvedPinkGhost(Sint32 iID) : BrainEvolved()
{
    string sNumber;

    // Out of range
    if(iID > PME_GA_POPULATION)
    {
        sName = "Pink-Unknown";
        return;
    }
    iNNInput = PME_ANN_GHOST_PINK_INPUT;
    iNNOutput = PME_ANN_GHOST_OUTPUT;

    // Evolved ID
    if(iID < 0)
    {
        sName = "Pink-" + sName;
        load();
    }
    // Training ID
    else
    {
        Main::Instance().ITool().intToStrDec(iID, sNumber);
        sName = "Pink-Training";
        sName = sName + "-" + sNumber;
    }
}
Sint32 BrainEvolvedPinkGhost::think(Actor* pActor, Sint32& iTX, Sint32& iTY, GameField* pGF)
{
    double dTmp;
    vector<double> vInputs, vOutputs;
    Sint32 iDirX, iDirY;
    
    // In case of problems with the neural network, fallback to fixed target
    if(pNeuralNet == nullptr) return 0;

    // Get PacMan position and direction
    pGF->getObjectPosition(PME_OBJECT_PACMAN, iTX, iTY);
    pGF->getObjectDirection(PME_OBJECT_PACMAN, iDirX, iDirY);
    
    // Prepare inputs
    // 4 input: PacMan (x,y), PacMan direction +4 -> normalization to [0,1]
    vInputs.clear();
    dTmp = ((double)iTX / (double)MAZE_WIDTH);
    vInputs.push_back(dTmp);
    dTmp = ((double)iTY / (double)MAZE_HEIGHT);
    vInputs.push_back(dTmp);
    dTmp = ((double)(iTX + 4 * iDirX) / (double)MAZE_HEIGHT);
    vInputs.push_back(dTmp);
    dTmp = ((double)(iTY + 4 * iDirY) / (double)MAZE_HEIGHT);
    vInputs.push_back(dTmp);

    // Call neural network
    vOutputs = pNeuralNet->update(vInputs);

    // Prepare outputs
    // 2 output from [0,1], convert it to target coordinates    
    if(!vOutputs.empty())
    {
        iTX = (Sint32)Main::Instance().ITool().round(vOutputs[0] * MAZE_WIDTH);
        iTY = (Sint32)Main::Instance().ITool().round(vOutputs[1] * MAZE_HEIGHT);
    }

    return 0;
}

// Blue Ghost brains
// Fixed brain
BrainBlueGhost::BrainBlueGhost() : Brain() { sName = "BlueGhost-Fixed"; }
Sint32 BrainBlueGhost::think(Actor* pActor, Sint32& iTX, Sint32& iTY, GameField* pGF) // Target PacMan position plus 2 where he is facing to and Red Ghost vector to it multiply x2
{
    Sint32 iDirX, iDirY, iGRX, iGRY;

    // Get current position and direction of PacMan
    pGF->getObjectPosition(PME_OBJECT_PACMAN, iTX, iTY);
    pGF->getObjectDirection(PME_OBJECT_PACMAN, iDirX, iDirY);

    // Calculate target (PacMain position plus 2 where he is facing)
    iTX = iTX + 2 * iDirX;
    iTY = iTY + 2 * iDirY;

    // Get current position of Red Ghost
    pGF->getObjectPosition(PME_OBJECT_GHOST_RED, iGRX, iGRY);

    // Set final target
    iTX = iGRX + 2 * (iTX - iGRX);
    iTY = iGRY + 2 * (iTY - iGRY);

    return 0;
}

// Evolved brain
BrainEvolvedBlueGhost::BrainEvolvedBlueGhost(Sint32 iID) : BrainEvolved()
{
    string sNumber;

    // Out of range
    if(iID > PME_GA_POPULATION)
    {
        sName = "Blue-Unknown";
        return;
    }
    iNNInput = PME_ANN_GHOST_BLUE_INPUT;
    iNNOutput = PME_ANN_GHOST_OUTPUT;

    // Evolved ID
    if(iID < 0)
    {
        sName = "Blue-" + sName;
        load();
    }
    // Training ID
    else
    {
        Main::Instance().ITool().intToStrDec(iID, sNumber);
        sName = "Blue-Training";
        sName = sName + "-" + sNumber;
    }
}
Sint32 BrainEvolvedBlueGhost::think(Actor* pActor, Sint32& iTX, Sint32& iTY, GameField* pGF)
{
    double dTmp;
    vector<double> vInputs, vOutputs;
    MazePoint pointRed;
    
    // In case of problems with the neural network, fallback to fixed target
    if(pNeuralNet == nullptr) return 0;

    // Get PacMan and Red Ghost positions and calculate Red-to-PacMan distance
    pGF->getObjectPosition(PME_OBJECT_PACMAN, iTX, iTY);
    pGF->getObjectPosition(PME_OBJECT_GHOST_RED, pointRed.iX, pointRed.iY);
    pActor->euclideanDistance(iTX, iTY, pointRed);

    // Prepare inputs
    // 3 input: PacMan (x,y), Red Ghost distance to PacMan -> normalization to [0,1]
    vInputs.clear();
    dTmp = ((double)iTX / (double)MAZE_WIDTH);
    vInputs.push_back(dTmp);
    dTmp = ((double)iTY / (double)MAZE_HEIGHT);
    vInputs.push_back(dTmp);
    dTmp = pointRed.dDistance / 38.0; // ~38 is the maximum distance on the maze -> sqrt(MAZE_WIDTH - 2)2 + (MAZE_HEIGHT - 2)2)
    vInputs.push_back(dTmp);

    // Call neural network
    vOutputs = pNeuralNet->update(vInputs);

    // Prepare outputs
    // 2 output from [0,1], convert it to target coordinates  
    if(!vOutputs.empty())
    {
        iTX = (Sint32)Main::Instance().ITool().round(vOutputs[0] * MAZE_WIDTH);
        iTY = (Sint32)Main::Instance().ITool().round(vOutputs[1] * MAZE_HEIGHT);
    }

    return 0;
}

// Orange Ghost brains
// Fixed brain
BrainOrangeGhost::BrainOrangeGhost() : Brain() { sName = "OrangeGhost-Fixed"; }
Sint32 BrainOrangeGhost::think(Actor* pActor, Sint32& iTX, Sint32& iTY, GameField* pGF) // Target PacMan when far from him 8 maze points or target scattering when close than 8
{
    Sint32 iPX, iPY;
    MazePoint pointActor;

    // Get PacMan position
    pGF->getObjectPosition(PME_OBJECT_PACMAN, iPX, iPY);

    // Get distance to PacMan
    pActor->getPositionMaze(pointActor.iX, pointActor.iY);
    pActor->euclideanDistance(iPX, iPY, pointActor);
    
    // Distance greater than 8, target PacMan
    if(pointActor.dDistance > 8)
    {
        iTX = iPX;
        iTY = iPY;
    }
    // Distance smaller than 8, target scattering position
    else reinterpret_cast<Ghost*>(pActor)->getScatteringTarget(iTX, iTY);        

    return 0;
}

// Evolved brain
BrainEvolvedOrangeGhost::BrainEvolvedOrangeGhost(Sint32 iID) : BrainEvolved()
{
    string sNumber;

    // Out of range
    if(iID > PME_GA_POPULATION)
    {
        sName = "Orange-Unknown";
        return;
    }
    iNNInput = PME_ANN_GHOST_ORANGE_INPUT;
    iNNOutput = PME_ANN_GHOST_OUTPUT;

    // Evolved ID
    if(iID < 0)
    {
        sName = "Orange-" + sName;
        load();
    }
    // Training ID
    else
    {
        Main::Instance().ITool().intToStrDec(iID, sNumber);
        sName = "Orange-Training";
        sName = sName + "-" + sNumber;
    }
}
Sint32 BrainEvolvedOrangeGhost::think(Actor* pActor, Sint32& iTX, Sint32& iTY, GameField* pGF)
{
    double dTmp;
    vector<double> vInputs, vOutputs;
    MazePoint pointActor;

    // In case of problems with the neural network, fallback to fixed target
    if(pNeuralNet == nullptr) return 0;

    // Get PacMan and our positions and calculate distance
    pGF->getObjectPosition(PME_OBJECT_PACMAN, iTX, iTY);
    pActor->getPositionMaze(pointActor.iX, pointActor.iY);
    pActor->euclideanDistance(iTX, iTY, pointActor);

    // Get position of the closest pellet to PacMan
    pGF->getClosestPellet(PME_OBJECT_PACMAN, iTX, iTY);

    // Prepare inputs
    // 3 input: Distance to PacMan and closest pellet to PacMan (x,y) -> normalization to [0,1]
    vInputs.clear();
    dTmp = ((double)iTX / (double)MAZE_WIDTH);
    vInputs.push_back(dTmp);
    dTmp = ((double)iTY / (double)MAZE_HEIGHT);
    vInputs.push_back(dTmp);
    dTmp = pointActor.dDistance / 38.0; // ~38 is the maximum distance on the maze -> sqrt(MAZE_WIDTH - 2)2 + (MAZE_HEIGHT - 2)2)
    vInputs.push_back(dTmp);

    // Call neural network
    vOutputs = pNeuralNet->update(vInputs);

    // Prepare outputs
    // 2 output from [0,1], convert it to target coordinates  
    if(!vOutputs.empty())
    {
        iTX = (Sint32)Main::Instance().ITool().round(vOutputs[0] * MAZE_WIDTH);
        iTY = (Sint32)Main::Instance().ITool().round(vOutputs[1] * MAZE_HEIGHT);
    }

    return 0;
}