/*----------------------------------------------------------------------
Pac-Man Evolution - Roberto Prieto
 Copyright (C) 2018-2025 MegaStorm Systems
contact@megastormsystems.com - http://www.megastormsystems.com

This software is provided 'as-is', without any express or implied
warranty.  In no event will the authors be held liable for any damages
arising from the use of this software.

Permission is granted to anyone to use this software for any purpose,
including commercial applications, and to alter it and redistribute it
freely, subject to the following restrictions:

1. The origin of this software must not be misrepresented; you must not
claim that you wrote the original software. If you use this software
in a product, an acknowledgment in the product documentation would be
appreciated but is not required.
2. Altered source versions must be plainly marked as such, and must not be
misrepresented as being the original software.
3. This notice may not be removed or altered from any source distribution.

------------------------------------------------------------------------

Artificial Neural Network class

Based on the amazing docs created by Mat Buckland (http://www.ai-junkie.com/ann/evolved/nnt1.html)

------------------------------------------------------------------------ */

#include "ArtificialNeuralNet.h"
#include <math.h>

// Function for generating weights inside the weight range
float generateWeights(float fMin, float fMax)
{
    return applyWeightPrecision((fMax - fMin) * ((float)Main::Instance().ITool().randWELL() / (float)4294967295) + fMin);
}
// Function for applyting our precision
float applyWeightPrecision(float fInput)
{
    return std::roundf(fInput / ANN_PRECISION) * ANN_PRECISION;
}

// Neuron constructor
Neuron::Neuron(Sint32 NumInputs)
{
    Sint32 i;
    Tool &mTool = Main::Instance().ITool();

    #ifdef ANN_ENABLE_BIAS
        // Additional weight for the bias
        iNumInputs = NumInputs + 1;
    #else
        iNumInputs = NumInputs;
    #endif
    
    for(i = 0; i < iNumInputs; ++i)
    {
        // Set up the weights with an initial random value between ANN_WEIGHT_MIN and ANN_WEIGHT_MAX
        vWeight.push_back(generateWeights());
    }
}

// NeuronLayer constructor
NeuronLayer::NeuronLayer(Sint32 NumNeurons,Sint32 NumInputsPerNeuron)
{
    Sint32 i;
    iNumNeurons = NumNeurons;

    for(i = 0; i < NumNeurons; ++i)
        vNeurons.push_back(Neuron(NumInputsPerNeuron));
}

// ArtificialNeuralNet constructor
ArtificialNeuralNet::ArtificialNeuralNet() 
{
    iActivationFunc = ANN_ACTIVATION_SIGMOID;
    iNumInputs = 0;
    iNumOutputs = 0;
    iNumHiddenLayers = 0;
    iNeuronsPerHiddenLyr = 0;
}

// Initialize the NN. It creates random weight from [ANN_WEIGHT_MIN, ANN_WEIGHT_MAX]
void ArtificialNeuralNet::init(Sint32 NumInputs, Sint32 NumOutputs, Sint32 NumHiddenLayers, Sint32 NeuronsPerHiddenLayer)
{
    iNumInputs = NumInputs;
    iNumOutputs = NumOutputs;
    iNumHiddenLayers =	NumHiddenLayers;
    iNeuronsPerHiddenLyr =	NeuronsPerHiddenLayer;

    // Create the layers
    if(iNumHiddenLayers > 0)
    {
        // First layer (input layer)
        vNeuronsLayers.push_back(NeuronLayer(iNeuronsPerHiddenLyr, iNumInputs));

        // Hidden layers
        for(Sint32 i = 0; i < iNumHiddenLayers - 1; ++i)
        {
            vNeuronsLayers.push_back(NeuronLayer(iNeuronsPerHiddenLyr,iNeuronsPerHiddenLyr));
        }

        // Output layer
        vNeuronsLayers.push_back(NeuronLayer(iNumOutputs, iNeuronsPerHiddenLyr));
    }

    else
    {
        // Only one layer: the input and output layer
        vNeuronsLayers.push_back(NeuronLayer(iNumOutputs, iNumInputs));
    }
}

// Set activation function
void ArtificialNeuralNet::setActivationFunction(Sint32 iNewActivation)
{
    if(iNewActivation <= ANN_ACTIVATION_LINEAR) iActivationFunc = ANN_ACTIVATION_LINEAR;
    else if(iNewActivation == ANN_ACTIVATION_STEP) iActivationFunc = ANN_ACTIVATION_STEP;
    else iActivationFunc = ANN_ACTIVATION_SIGMOID;
}

// Get activation function
Sint32 ArtificialNeuralNet::getActivationFunction()
{
    return iActivationFunc;
}

// Get a vector with all the weights
vector<double> ArtificialNeuralNet::getWeights()
{
    vector<double> weights;
    Sint32 i, j, k;

    // Loop through each layer
    for(i = 0; i < iNumHiddenLayers + 1; ++i)
    {
        // Neurons
        for(j = 0; j < vNeuronsLayers[i].iNumNeurons; ++j)
        {
            // and weights
            for(k = 0; k < vNeuronsLayers[i].vNeurons[j].iNumInputs; ++k)
            {
                weights.push_back(vNeuronsLayers[i].vNeurons[j].vWeight[k]);
            }
        }
    }

    return weights;
}

// Set all the weights replacing current values using the provided vector<double>
Sint32 ArtificialNeuralNet::setWeights(vector<double>& vW)
{
    Sint32 cWeight = 0, i, j, k;

    // Check we have enough weights
    if(getNumberOfWeights() > vW.size()) return -1;
        
    // Loop through each layer
    for(i = 0; i < iNumHiddenLayers + 1; ++i)
    {
        // Neurons
        for(j = 0; j < vNeuronsLayers[i].iNumNeurons; ++j)
        {
            // and weights
            for (k = 0; k < vNeuronsLayers[i].vNeurons[j].iNumInputs; ++k)
            {
                vNeuronsLayers[i].vNeurons[j].vWeight[k] = vW[cWeight++];
            }
        }
    }

    return 0;
}

// Return the number of weights
Sint32 ArtificialNeuralNet::getNumberOfWeights()
{
    Sint32 i, j, weights = 0;

    // Loop through each layer
    for(i = 0; i < iNumHiddenLayers + 1; ++i)
    {
        // Neurons
        for(j = 0; j < vNeuronsLayers[i].iNumNeurons; ++j)
        {
            weights += vNeuronsLayers[i].vNeurons[j].iNumInputs; 			
        }
    }

    return weights;
}

// Using the input vector, it calculates using the NN the returned output vector
vector<double> ArtificialNeuralNet::update(vector<double>& inputs)
{
    vector<double> outputs;
    Sint32 cWeight = 0, i, j, k;

    // We need a match between our inputs and the size of the input vector
    if(inputs.size() != iNumInputs)
    {
        // Return empty vector
        return outputs;
    }

    // Loop through each layer
    for(i = 0; i < iNumHiddenLayers + 1; ++i)
    {		
        if(i > 0)
        {
            inputs = outputs;
        }

        outputs.clear();		
        cWeight = 0;

        // For each neuron sum the (inputs * corresponding weights). 
        for(j = 0; j < vNeuronsLayers[i].iNumNeurons; ++j)
        {
            double netinput = 0;

            Sint32	NumInputs = vNeuronsLayers[i].vNeurons[j].iNumInputs;

            // For each weight
            #ifdef ANN_ENABLE_BIAS
                for(k = 0; k < NumInputs - 1; ++k)
            #else
                for(k = 0; k < NumInputs; ++k)
            #endif
            {
                // Sum the weights x inputs
                netinput += vNeuronsLayers[i].vNeurons[j].vWeight[k] * inputs[cWeight++];
            }

            #ifdef ANN_ENABLE_BIAS
                // Add in the bias (-1.0)
                netinput += vNeuronsLayers[i].vNeurons[j].vWeight[NumInputs - 1] * (-1.0);
            #endif

            // The result is sent to our activation function and stored in the output vector
            switch(iActivationFunc)
            {
            case ANN_ACTIVATION_LINEAR:
                outputs.push_back(linear(netinput, 1.0));
                break;
            case ANN_ACTIVATION_STEP:
                outputs.push_back(step(netinput, 1.0));
                break;
            case ANN_ACTIVATION_SIGMOID:
                outputs.push_back(sigmoid(netinput, 1.0));
                break;
            }
            cWeight = 0;
        }
    }

    return outputs;
}

// Get neural network components
Sint32 ArtificialNeuralNet::getNumOutputs()
{
    return iNumOutputs;
}
Sint32 ArtificialNeuralNet::getNumInputs()
{
    return iNumInputs;
}
Sint32 ArtificialNeuralNet::getNumHiddenLayers()
{
    return iNumHiddenLayers;
}
Sint32 ArtificialNeuralNet::getNumNeuronsPerLayer()
{
    return iNeuronsPerHiddenLyr;
}

// Activation function: Sigmoid
double ArtificialNeuralNet::sigmoid(double netinput, double response)
{
    return( 1 / ( 1 + exp(-netinput / response)));
}

// Activation function: Linear
double ArtificialNeuralNet::linear(double netinput, double scale)
{
    return netinput * scale;

}

// Activation function: Step
double ArtificialNeuralNet::step(double netinput, double threshold)
{
    if(netinput > threshold) return 1;
    else return 0;
}
